#include <PlainSession.h>

#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>

namespace MetadataExporter_ns
{

//==============================================================================
//      PlainSession::PlainSession()
//==============================================================================
PlainSession::PlainSession(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp,
    boost::shared_ptr<boost::asio::io_service> ioService_sp) :
    Session::Session(deviceImpl_p,configuration_sp, dBManager_sp, ioService_sp),
    m_plainSocket(*ioService_sp)
{
    DEBUG_STREAM << "PlainSession::PlainSession()" << endl;
}

//==============================================================================
//      PlainSession::~PlainSession()
//==============================================================================
PlainSession::~PlainSession()
{
    DEBUG_STREAM << "PlainSession::~PlainSession()" << endl;

    INFO_STREAM << "PlainSession::~PlainSession() Disconnection from "
        << m_remoteEndpoint << endl;

    boost::system::error_code errorCode;

    m_plainSocket.shutdown(boost::asio::ip::tcp::socket::shutdown_both, errorCode);

    m_plainSocket.close(errorCode);
}

//==============================================================================
//      PlainSession::create()
//==============================================================================
Session::SP PlainSession::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp,
    boost::shared_ptr<boost::asio::io_service> ioService_sp)
{
    Session::SP s_sp(new PlainSession(deviceImpl_p, configuration_sp,
    dBManager_sp, ioService_sp), PlainSession::Deleter());

    return s_sp;
}

//==============================================================================
//      PlainSession::getSocket()
//==============================================================================
boost::asio::ip::tcp::socket& PlainSession::getSocket()
{
    DEBUG_STREAM << "PlainSession::getSocket()" << endl;

    return m_plainSocket;
}

//==============================================================================
//      PlainSession::start()
//==============================================================================
void PlainSession::start()
{
    DEBUG_STREAM << "PlainSession::start()" << endl;

    m_remoteEndpoint = boost::lexical_cast<std::string>(
        m_plainSocket.remote_endpoint());

    INFO_STREAM << "PlainSession::start() Connection from "
        << m_remoteEndpoint << endl;

    m_protocolManager_sp->setRemoteEndpoint(m_remoteEndpoint);

    startReadRequestHeader();
}

//==============================================================================
//      PlainSession::startReadRequestHeader()
//==============================================================================
void PlainSession::startReadRequestHeader()
{
        DEBUG_STREAM << "PlainSession::startReadRequestHeader()" << endl;

        m_readBuff.resize(HEADER_SIZE);

        boost::asio::async_read(m_plainSocket, boost::asio::buffer(m_readBuff),
            m_strand.wrap(boost::bind(&PlainSession::handleReadRequestHeader,
            shared_from_this(), boost::asio::placeholders::error)));
}

//==============================================================================
//      PlainSession::startReadRequestBody()
//==============================================================================
void PlainSession::startReadRequestBody(boost::uint32_t bodySize)
{
    DEBUG_STREAM << "PlainSession::startReadRequestBody()" << endl;

    m_readBuff.resize(HEADER_SIZE + bodySize);

    boost::asio::mutable_buffers_1 mutableBuffer =
        boost::asio::buffer(&m_readBuff[HEADER_SIZE], bodySize);

    #ifdef VERBOSE_DEBUG
        INFO_STREAM << "PlainSession::startReadRequestBody() "
            << m_remoteEndpoint << " >>>> " << bodySize << " BYTE" << endl;
    #endif

    boost::asio::async_read(m_plainSocket, mutableBuffer,
        m_strand.wrap(boost::bind(&PlainSession::handleReadRequestBody,
        shared_from_this(), boost::asio::placeholders::error)));
}

//==============================================================================
//      PlainSession::startWriteResponse()
//==============================================================================
void PlainSession::startWriteResponse()
{
    DEBUG_STREAM << "PlainSession::startWriteResponse()" << endl;

    try
    {
        RequestSP request_sp(new Request);

        request_sp->ParseFromArray(&m_readBuff[HEADER_SIZE], m_readBuff.size() - HEADER_SIZE);

        ResponseSP response_sp = m_protocolManager_sp->prepareResponse(request_sp);

        boost::uint32_t bodySize = response_sp->ByteSize();

        std::vector<boost::uint8_t> writeBuff;
        writeBuff.resize(HEADER_SIZE + bodySize);

        encodeHeader(writeBuff, bodySize);

        response_sp->SerializeToArray(&writeBuff[HEADER_SIZE], bodySize);

        #ifdef VERBOSE_DEBUG
            INFO_STREAM << "PlainSession::startWriteResponse() "
                << m_remoteEndpoint << " <<<< " << bodySize << " byte" << endl;
        #endif

        boost::asio::async_write(m_plainSocket, boost::asio::buffer(writeBuff),
            m_strand.wrap(boost::bind(&PlainSession::handleWriteResponse,
                shared_from_this(), boost::asio::placeholders::error)));
    }
    catch(std::runtime_error& ec)
    {
        ERROR_STREAM << "SSLSession::startWriteResponse() "
            << ec.what() << " from " << m_remoteEndpoint << endl;
    }
    catch(...)
    {
        ERROR_STREAM << "SSLSession::startWriteResponse() unknown error from "
            << m_remoteEndpoint << endl;

    }
}

}   //namespace