#include <ProtocolManager.h>

namespace MetadataExporter_ns
{

//==============================================================================
//      ProtocolManager::ProtocolManager()
//==============================================================================
ProtocolManager::ProtocolManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp) :
    Tango::LogAdapter(deviceImpl_p), m_configuration_sp(configuration_sp),
    m_dBManager_sp(dBManager_sp)
{
    DEBUG_STREAM << "ProtocolManager::ProtocolManager()" << endl;

    m_isAuthorised = false;
    m_isValidated = false;
}

//==============================================================================
//      ProtocolManager::~ProtocolManager()
//==============================================================================
ProtocolManager::~ProtocolManager()
{
    DEBUG_STREAM << "ProtocolManager::~ProtocolManager()" << endl;
}

//==============================================================================
//      ProtocolManager::create()
//==============================================================================
ProtocolManager::SP ProtocolManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp, DBManager::SP dBManager_sp)
{
    ProtocolManager::SP d_sp(new ProtocolManager(deviceImpl_p, configuration_sp,
        dBManager_sp), ProtocolManager::Deleter());

    return d_sp;
}

//==============================================================================
//      ProtocolManager::setRemoteEndpoint()
//==============================================================================
void ProtocolManager::setRemoteEndpoint(std::string remoteEndpoint)
{
    DEBUG_STREAM << "ProtocolManager::setRemoteEndpoint()" << endl;

    m_remoteEndpoint = remoteEndpoint;
}

//==============================================================================
//      ProtocolManager::prepareResponse()
//==============================================================================
ResponseSP ProtocolManager::prepareResponse(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareResponse()" << endl;

    ResponseSP response_sp;

    switch(request_sp->type())
    {
        case Request::AUTHORIZATION:
        {
            response_sp = prepareAuthroisation(request_sp);
            break;
        }
        case Request::VALIDATION:
        {
            response_sp = prepareValidation(request_sp);
            break;
        }
        case Request::METADATA:
        {
            response_sp = prepareMetadata(request_sp);
            break;
        }
        default:
            throw std::runtime_error("Unknown request type!");
    }

    if(!response_sp->IsInitialized())
        throw std::runtime_error("Not initialized response!");

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareAuthroisation()
//==============================================================================
ResponseSP ProtocolManager::prepareAuthroisation(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareAuthroisation()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::AUTHORIZATION);

    Response::Authorization* auth_resp = response_sp->mutable_authorization();

    if(!m_isAuthorised)
    {
        const Request::Authorization& auth_req = request_sp->authorization();
        std::string username =  auth_req.username();
        std::string password = auth_req.password();

        if(m_configuration_sp->isUserAuthorized(username, password))
        {
            INFO_STREAM << "ProtocolManager::prepareAuthroisation() "
                << "Authorization accepted from " << m_remoteEndpoint << endl;

            m_isAuthorised = true;

            auth_resp->set_state(Response::Authorization::ACCEPTED);
            auth_resp->set_status("Authorization accepted");
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareAuthroisation() "
                << "Invalid username or password from " << m_remoteEndpoint << endl;

            m_isAuthorised = false;

            auth_resp->set_state(Response::Authorization::REJECTED);
            auth_resp->set_status("Invalid username or password");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareAuthroisation() "
            << "Already authorized from " << m_remoteEndpoint << endl;

        auth_resp->set_state(Response::Authorization::REJECTED);
        auth_resp->set_status("Already authorized");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareValidation()
//==============================================================================
ResponseSP ProtocolManager::prepareValidation(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareValidation()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::VALIDATION);

    Response::Validation* validationRes = response_sp->mutable_validation();

    if(m_isAuthorised)
    {
        if(!m_isValidated)
        {
            const Request::Validation& validationReq = request_sp->validation();

            m_validatedSchema = validationReq.schema();
            m_validatedTable =  validationReq.table();

            try
            {
                DBManager::InformationList informationList =
                    m_dBManager_sp->retrieveInformation(m_validatedSchema, m_validatedTable);

                if(validationReq.columns_size() != (int)informationList.size())
                    throw std::runtime_error("Columns number does not match");

                const google::protobuf::RepeatedPtrField
                    < Request::Validation::Column >& columns = validationReq.columns();

                google::protobuf::RepeatedPtrField
                    < Request::Validation::Column >::const_iterator it;

                for(it=columns.begin(); it!=columns.end(); ++it)
                    validateColumn(*it, informationList);

                INFO_STREAM << "ProtocolManager::prepareValidation() "
                    << "Validation accepted for " << m_validatedSchema << "."
                    << m_validatedTable << " from " << m_remoteEndpoint << endl;

                m_isValidated = true;

                validationRes->set_state(Response::Validation::ACCEPTED);
                validationRes->set_status("Table validated");
            }
            catch(std::runtime_error& ex)
            {
                WARN_STREAM << "ProtocolManager::prepareValidation() "
                    << ex.what() << " from " << m_remoteEndpoint << endl;

                validationRes->set_state(Response::Validation::REJECTED);
                validationRes->set_status(ex.what());
            }
            catch(...)
            {
                WARN_STREAM << "ProtocolManager::prepareValidation() "
                    << "Unknown exception from " << m_remoteEndpoint << endl;

                validationRes->set_state(Response::Validation::REJECTED);
                validationRes->set_status("Unknown exception");
            }
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareValidation() "
                << "Already validated from " << m_remoteEndpoint << endl;

            validationRes->set_state(Response::Validation::REJECTED);
            validationRes->set_status("Already validated");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareValidation() "
            << "Not authorised from " << m_remoteEndpoint << endl;

        validationRes->set_state(Response::Validation::REJECTED);
        validationRes->set_status("Not authorised");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::prepareMetadata()
//==============================================================================
ResponseSP ProtocolManager::prepareMetadata(RequestSP request_sp)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::prepareMetadata()" << endl;

    ResponseSP response_sp(new Response());

    response_sp->set_type(Response::METADATA);

    Response::Metadata* metadataRes = response_sp->mutable_metadata();

    if(m_isAuthorised)
    {
        if(m_isValidated)
        {
            const Request::Metadata& metadataReq = request_sp->metadata();

            int64_t rawTimestamp = metadataReq.timestamp();
            std::tm tmTimestamp = *localtime(&rawTimestamp);

            DEBUG_STREAM << "ProtocolManager::prepareMetadata() Searching in "
                << m_validatedSchema << "." << m_validatedTable << " timestamp "
                << asctime(&tmTimestamp) << " from " << m_remoteEndpoint << endl;

            try
            {
                if(!m_rowSet_sp)
                {
                    m_rowSet_sp = m_dBManager_sp->retrieveNewTuples(
                        m_validatedSchema, m_validatedTable, tmTimestamp);

                    m_it = m_rowSet_sp->begin();
                }

                if(m_rowSet_sp && m_it != m_rowSet_sp->end())
                {
                    metadataRes->set_state(Response::Metadata::ACCEPTED);
                    metadataRes->set_status("Data ready");

                    fillMetadata(metadataRes);
                }
                else
                {
                    metadataRes->set_state(Response::Metadata::ACCEPTED);
                    metadataRes->set_status("No more data");
                }
            }
            catch(std::runtime_error& ex)
            {
                WARN_STREAM << "ProtocolManager::prepareMetadata() "
                    << ex.what() << " from " << m_remoteEndpoint << endl;

                metadataRes->set_state(Response::Metadata::REJECTED);
                metadataRes->set_status(ex.what());

                m_rowSet_sp.reset();
            }
            catch(...)
            {
                WARN_STREAM << "ProtocolManager::prepareMetadata() "
                    << "Unknown exception from " << m_remoteEndpoint << endl;

                metadataRes->set_state(Response::Metadata::REJECTED);
                metadataRes->set_status("Unknown exception");

                m_rowSet_sp.reset();
            }
        }
        else
        {
            WARN_STREAM << "ProtocolManager::prepareMetadata() "
                << "Not validated from " << m_remoteEndpoint << endl;

            metadataRes->set_state(Response::Metadata::REJECTED);
            metadataRes->set_status("Not validated");
        }
    }
    else
    {
        WARN_STREAM << "ProtocolManager::prepareMetadata() "
            << "Not authorised from " << m_remoteEndpoint << endl;

        metadataRes->set_state(Response::Metadata::REJECTED);
        metadataRes->set_status("Not authorised");
    }

    return response_sp;
}

//==============================================================================
//      ProtocolManager::validateColumn()
//==============================================================================
void ProtocolManager::validateColumn(const Request::Validation::Column& column,
    DBManager::InformationList& informationList) throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::validateColumn()" << endl;

    bool found = false;

    DBManager::InformationList::const_iterator it;
    for(it=informationList.begin(); it!=informationList.end(); ++it)
    {
        if(!it->get<0>())
            throw std::runtime_error("Empty column name");
        std::string columnName = it->get<0>().get();

        if(column.name().compare(columnName)==0)
        {
            found = true;

            if(!it->get<1>())
                throw std::runtime_error("Empty column type");
            std::string columnType = it->get<1>().get();

            if(column.type().compare(columnType)!=0)
            {
                std::stringstream errorStream;
                errorStream << "Column " << column.name() << " type error "
                    << "server " << columnType << " client " << column.type();
                throw std::runtime_error(errorStream.str());
            }

            if(!it->get<2>())
                throw std::runtime_error("Empty is nullable");
            std::string isNullable = it->get<2>().get();

            if(column.nullable().compare(isNullable)!=0)
            {
                std::stringstream errorStream;
                errorStream << "Column " << column.name() << " nullable error "
                    << "server " << isNullable << " client " << column.nullable();
                throw std::runtime_error(errorStream.str());
            }

            #ifdef VERBOSE_DEBUG
                DEBUG_STREAM << "CLIENT: " << columnName << " | " << columnType
                    <<  " | " << isNullable << endl;
                DEBUG_STREAM << "SERVER: " << column.name() << " | "
                    << column.type() <<  " | " << column.nullable() << endl;
            #endif
        }
    }

    if(!found)
    {
        std::stringstream errorStream;
        errorStream << "Column " << column.name() << " not found on server";
        throw std::runtime_error(errorStream.str());
    }
}

//==============================================================================
//      ProtocolManager::fillMetadata()
//==============================================================================
void ProtocolManager::fillMetadata(Response::Metadata* metadataRes)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::fillMetadata()" << endl;

    std::time_t currentTime = 0;

    while(m_it != m_rowSet_sp->end())
    {
        int id = m_it->get<int>("id");
        std::string file_path = m_it->get<std::string>("file_path");
        int file_version = m_it->get<int>("file_version");
        std::string file_name = m_it->get<std::string>("file_name");
        std::tm newTm = m_it->get<std::tm>("update_time");

        #ifdef VERBOSE_DEBUG
            INFO_STREAM << "|" << id << "|" << file_path << "|" << file_version
                << "|" << file_name << "|" << asctime(&newTm) << "|" << endl;
        #endif

        std::time_t newTime = mktime(&newTm);

        if(currentTime == 0)
            currentTime = newTime;

        std::tm currentTm = *localtime(&currentTime);

        INFO_STREAM << "NEW TIME: " << asctime(&newTm) << endl;
        INFO_STREAM << "CURRENT TIME: " << asctime(&currentTm) << endl;

        if(difftime(newTime, currentTime)>0)
        {
            INFO_STREAM << "BREAK" << endl;
            break;
        }

        Response::Metadata::Row* row = metadataRes->add_rows();
        fillRow(row);

        m_it++;
    }

    if(m_it == m_rowSet_sp->end())
    {
        INFO_STREAM << "RESET" << endl;
        m_rowSet_sp.reset();
    }
}

//==============================================================================
//      ProtocolManager::fillRow()
//==============================================================================
void ProtocolManager::fillRow(Response::Metadata::Row* row)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ProtocolManager::fillRow()" << endl;

    for(std::size_t i=0; i!=m_it->size(); ++i)
    {
        const soci::column_properties& props = m_it->get_properties(i);

        std::string name = props.get_name();

        if(m_it->get_indicator(i) == soci::i_null)
        {
            DEBUG_STREAM << "NAME: " << name << " NULL" << endl;
            continue;
        }

        switch(props.get_data_type())
        {
            case soci::dt_string:
            {
                std::string value = m_it->get<std::string>(i);

                DEBUG_STREAM << "NAME: " << name << " " << value << endl;

                Response::Metadata::Row::DtString* dtString = row->add_strings_list();
                dtString->set_key(name);
                dtString->set_value(value);

                break;
            }
            case soci::dt_double:
            {
                double value = m_it->get<double>(i);

                DEBUG_STREAM << "NAME: " << name << " VALUE: " << value << endl;

                Response::Metadata::Row::DtDouble* dtDouble = row->add_double_list();
                dtDouble->set_key(name);
                dtDouble->set_value(value);

                break;
            }
            case soci::dt_integer:
            {
                int value = m_it->get<int>(i);

                DEBUG_STREAM << "NAME: " << name << " VALUE: " << value << endl;

                Response::Metadata::Row::DtInteger* dtInteger = row->add_integer_list();
                dtInteger->set_key(name);
                dtInteger->set_value(value);

                break;
            }
            case soci::dt_long_long:
            {
                long long value = m_it->get<long long>(i);

                DEBUG_STREAM << "NAME: " << name << " VALUE: " << value << endl;

                Response::Metadata::Row::DtLongLong* dtLongLong = row->add_long_long_list();
                dtLongLong->set_key(name);
                dtLongLong->set_value(value);

                break;
            }
            case soci::dt_unsigned_long_long:
            {
                unsigned long value = m_it->get<unsigned long>(i);

                DEBUG_STREAM << "NAME: " << name << " VALUE: " << value << endl;

                Response::Metadata::Row::DtUnsignedLong* dtUnsignedLong = row->add_unsinged_long_list();
                dtUnsignedLong->set_key(name);
                dtUnsignedLong->set_value(value);

                break;
            }
            case soci::dt_date:
            {
                std::tm value = m_it->get<std::tm>(i);

                DEBUG_STREAM << "NAME: " << name << " VALUE: " << asctime(&value) << endl;

                Response::Metadata::Row::DtDate* dtDate = row->add_date_list();
                dtDate->set_key(name);
                dtDate->set_value(mktime(&value));

                break;
            }
            default:
                throw std::runtime_error("Unknown data type");
        }
    }
}

}   //namespace
