#include <DBManager.h>

#include <boost/date_time.hpp>

#include <soci/mysql/soci-mysql.h>
#include <soci/use.h>

namespace MetadataExporter_ns
{

//==============================================================================
//      DBManager::DBManager()
//==============================================================================
DBManager::DBManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp) : Tango::LogAdapter(deviceImpl_p),
    m_configuration_sp(configuration_sp)
{
    DEBUG_STREAM << "DBManager::DBManager()" << endl;

    unsigned int connectionNumber = m_configuration_sp->getDatabaseConnectionNumber();

    m_connectionPool_sp.reset(new soci::connection_pool(connectionNumber));
}

//==============================================================================
//      DBManager::DBManager()
//==============================================================================
DBManager::~DBManager()
{
    DEBUG_STREAM << "DBManager::~DBManager()" << endl;
}

//==============================================================================
//      DBManager::DBManager()
//==============================================================================
DBManager::SP DBManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp)
{
    DBManager::SP d_sp(new DBManager(deviceImpl_p, configuration_sp),
        DBManager::Deleter());

    return d_sp;
}

//==============================================================================
//      DBManager::connect()
//==============================================================================
void DBManager::connect() throw(soci::soci_error)
{
    DEBUG_STREAM << "DBManager::connect()" << endl;

    boost::mutex::scoped_lock lock(m_connectionPoolMutex);

    std::stringstream connection;
    connection << " host=" << m_configuration_sp->getDatabaseHost();
    connection << " port=" << m_configuration_sp->getDatabasePort();
    connection << " user=" << m_configuration_sp->getDatabaseUsername();
    connection << " password=" << m_configuration_sp->getDatabasePassword();

    #ifdef VERBOSE_DEBUG
        INFO_STREAM << "DBManager::connect(): " << connection.str() << endl;
    #endif

    unsigned int connectionNumber = m_configuration_sp->getDatabaseConnectionNumber();

    for(unsigned int i=0; i<connectionNumber; ++i)
    {
        m_connectionPool_sp->at(i).open(soci::mysql, connection.str());
    }
}

//==============================================================================
//      DBManager::disconnect()
//==============================================================================
void DBManager::disconnect()
{
    DEBUG_STREAM << "DBManager::disconnect()" << endl;

    boost::mutex::scoped_lock lock(m_connectionPoolMutex);

    unsigned int connectionNumber = m_configuration_sp->getDatabaseConnectionNumber();

    for(unsigned int i=0; i<connectionNumber; ++i)
    {
        m_connectionPool_sp->at(i).close();
    }
}

//==============================================================================
//      DBManager::retrieveInformation()
//==============================================================================
DBManager::InformationList DBManager::retrieveInformation(std::string schema,
    std::string table) throw(soci::soci_error)
{
    DEBUG_STREAM << "DBManager::retrieveInformation()" << endl;

    soci::session session(*m_connectionPool_sp);

    if(session.get_backend() == NULL)
        session.reconnect();

    soci::rowset<InformationTuple> rows = (session.prepare << "select "
        "column_name, column_type, is_nullable from information_schema.columns "
        "where table_schema like :schema and table_name like :table",
        soci::use(schema, "schema"), soci::use(table, "table"));

    InformationList informationList;

    std::copy(rows.begin(), rows.end(), std::back_inserter(informationList));

    if(informationList.empty())
    {
        std::stringstream errorStream;
        errorStream << schema << "." << table << " not exists";
        throw soci::soci_error(errorStream.str());
    }

    return informationList;
}

//==============================================================================
//      DBManager::retrieveNewTuples()
//==============================================================================
DBManager::RowsetSP DBManager::retrieveNewTuples(std::string schema,
    std::string table, std::tm update_time) throw(soci::soci_error)
{
    DEBUG_STREAM << "DBManager::retrieveNewTuples()" << endl;

    soci::session session(*m_connectionPool_sp);

    if(session.get_backend() == NULL)
        session.reconnect();

    //@workaround: this does not work: timestamp parameter not passed
//    RowsetSP rows(new soci::rowset<soci::row>((session.prepare << "select * from "
//        << schema << "." << table << " where update_time > :timestamp "
//        << "order by update_time asc", soci::use(update_time,"timestamp"))));

    boost::posix_time::ptime timestamp = boost::posix_time::ptime_from_tm(update_time);

    RowsetSP rows(new soci::rowset<soci::row>(session.prepare << "select * from "
        << schema << "." << table << " where update_time>'"
        << boost::posix_time::to_iso_string(timestamp) << "' order by update_time asc"));

    return rows;
}

}   //namespace
