#include <ScriptManager.h>

namespace PreProcessor_ns
{

//==============================================================================
//	ScriptManager::ScriptManager()
//==============================================================================
ScriptManager::ScriptManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp) : Tango::LogAdapter(deviceImpl_p),
    m_configuration_sp(configuration_sp)
{
    DEBUG_STREAM << "ScriptManager::ScriptManager()" << endl;
}

//==============================================================================
//	ScriptManager::~ScriptManager()
//==============================================================================
ScriptManager::~ScriptManager()
{
    DEBUG_STREAM << "ScriptManager::~ScriptManager()" << endl;
}

//==============================================================================
//	EventBuffer::create()
//==============================================================================
ScriptManager::SP ScriptManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp)
{
    ScriptManager::SP s_sp(new ScriptManager(deviceImpl_p, configuration_sp),
        ScriptManager::Deleter());

    return s_sp;
}

//==============================================================================
//	ScriptManager::isReadyToArchive()
//==============================================================================
void ScriptManager::checkScriptCompliance()
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ScriptManager::checkScriptCompliance()" << endl;

    boost::filesystem::path scriptPath(m_configuration_sp->getScriptPath());

    if(!boost::filesystem::exists(scriptPath))
        throw std::runtime_error("Script path not exists");

    if(!boost::filesystem::is_regular_file(scriptPath))
        throw std::runtime_error("Script path is not regular file");

    std::stringstream command;
    command << m_configuration_sp->getScriptPath() << " CHECK";

    std::string result = exec(command.str());

    if(result.find("CHECK OK") == std::string::npos)
        throw std::runtime_error("Invalid script");
}

//==============================================================================
//	ScriptManager::isFileVerified()
//==============================================================================
bool ScriptManager::isFileVerified(boost::filesystem::path& filePath)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ScriptManager::isFileVerified()" << endl;

    std::stringstream command;
    command << m_configuration_sp->getScriptPath()
        << " VERIFY " << filePath.string();

    std::string result = exec(command.str());

    if(result.find("VERIFY OK") != std::string::npos)
    {
        return true;
    }
    else if(result.find("VERIFY WAIT") != std::string::npos)
    {
        return false;
    }
    else if(result.find("VERIFY FATAL") != std::string::npos)
    {
        std::stringstream errorStream;
        errorStream << "Verification error: " << result;
        throw std::runtime_error(errorStream.str());
    }
    else
    {
        std::stringstream errorStream;
        errorStream << "Unknown verification error: " << result;
        throw std::runtime_error(errorStream.str());
    }
}

//==============================================================================
//	ScriptManager::preProcessFile()
//==============================================================================
void ScriptManager::preProcessFile(boost::filesystem::path& filePath)
    throw(std::runtime_error)
{
    DEBUG_STREAM << "ScriptManager::preProcessFile()" << endl;

    std::stringstream command;
    command << m_configuration_sp->getScriptPath()
        << " PROCESS " << filePath.string();

    std::string result = exec(command.str());

    if(result.find("PROCESS OK") == std::string::npos)
    {
        if(result.find("PROCESS FATAL") != std::string::npos)
        {
            std::stringstream errorStream;
            errorStream << "Pre process error: " << result;
            throw std::runtime_error(errorStream.str());
        }
        else
        {
            std::stringstream errorStream;
            errorStream << "Unknown pre process error: " << result;
            throw std::runtime_error(errorStream.str());
        }
    }
}

//==============================================================================
//	ScriptManager::exec()
//==============================================================================
std::string ScriptManager::exec(std::string command) throw(std::runtime_error)
{
    DEBUG_STREAM << "ScriptManager::exec() << " << command << endl;

	const int BUFFSIZE = 1024;

    FILE* pipe = popen(command.c_str(), "r");

    if (!pipe)
    	throw std::runtime_error("Error launching external script");

    char buffer[BUFFSIZE];
    std::string result = "";

    while(!feof(pipe))
    {
        if(fgets(buffer, BUFFSIZE, pipe) != NULL)
                result += buffer;
    }

    pclose(pipe);

	DEBUG_STREAM << "ScriptManager::exec() >> " << result << endl;

    return result;
}

}


