#include <WorkerThread.h>
#include <PreProcessor.h>

#include <tango.h>

#include <boost/chrono.hpp>

namespace PreProcessor_ns
{

//==============================================================================
//	WorkerThread::WorkerThread()
//==============================================================================
WorkerThread::WorkerThread(PreProcessor* preProcessor_p,
    EventBuffer::SP eventBuffer_sp, ScriptManager::SP fileManager_sp,
    Configuration::SP configuration_sp) : Tango::LogAdapter(preProcessor_p),
    m_preProcessor_p(preProcessor_p), m_eventBuffer_sp(eventBuffer_sp),
    m_fileManager_sp(fileManager_sp), m_configuration_sp(configuration_sp)
{
	DEBUG_STREAM << "WorkerThread::WorkerThread()" << std::endl;
}

//==============================================================================
//	WorkerThread::~WorkerThread()
//==============================================================================
WorkerThread::~WorkerThread()
{
	DEBUG_STREAM << "WorkerThread::~WorkerThread()" << std::endl;
}

//==============================================================================
//	WorkerThread::workerLoop()()
//==============================================================================
void WorkerThread::workerLoop()
{
    DEBUG_STREAM << "WorkerThread::workerLoop(): starting loop" << endl;

    //Sleep time between two verification attempts
    boost::chrono::seconds sleepTime(m_configuration_sp->getSleepTime());

    //Verification attempts timeout
    boost::chrono::steady_clock::duration waitTime =
        boost::chrono::seconds(m_configuration_sp->getWaitTime());

    while(true)
    {
        try
        {
            //Wait new file event to process
            boost::filesystem::path origPath(m_eventBuffer_sp->waitNew());

            std::string fileName = origPath.stem().string();

            //Start timer
            boost::chrono::steady_clock::time_point start =
                boost::chrono::steady_clock::now();

            INFO_STREAM << "WorkerThread::workerLoop() processing \""
                << fileName << "\"" << endl;
            try
            {
                bool verified = false;

                do
                {
                    if(m_fileManager_sp->isFileVerified(origPath))
                    {
                        verified = true;
                        break;
                    }
                    else
                    {
                        boost::this_thread::sleep_for(sleepTime);
                    }
                }
                while(boost::chrono::steady_clock::now()-start <= waitTime);

                m_fileManager_sp->preProcessFile(origPath);

                copyToDestination(origPath);

                if(verified)
                {
                    INFO_STREAM << "WorkerThread::workerLoop() \"" << fileName
                        << "\" ingested regularly" << endl;
                    m_preProcessor_p->incrementRegularCounter();
                }
                else
                {
                    WARN_STREAM << "WorkerThread::workerLoop() \"" << fileName
                        << "\" archived after timeout" << endl;
                    m_preProcessor_p->incrementWarningCounter();
                }
            }
            catch(std::runtime_error& ex)
            {
                ERROR_STREAM << "WorkerThread::workerLoop() \"" << fileName
                    << "\" not archived due to " << ex.what() << endl;
                m_preProcessor_p->incrementErrorCounter();
            }

            m_eventBuffer_sp->markAsProcessed(origPath);
        }
		catch(boost::thread_interrupted& ex)
		{
            DEBUG_STREAM << "WorkerThread::workerLoop() interrupt" << endl;
			break;
		}
        catch(std::exception& ex)
        {
            ERROR_STREAM << "WorkerThread::workerLoop() " << ex.what() << endl;
        }
		catch(...)
		{
            ERROR_STREAM << "WorkerThread::workerLoop() unknown exception" << endl;
		}
    } //thread loop
}

//==============================================================================
//	WorkerThread::moveFile()()
//==============================================================================
void WorkerThread::copyToDestination(boost::filesystem::path& origPath)
    throw (std::runtime_error)
{
    DEBUG_STREAM << "WorkerThread::moveFile()" << endl;

	if(!boost::filesystem::is_regular_file(origPath))
		throw std::runtime_error( "Origin path \"" +
			origPath.string() + "\" is not a regular file");

    boost::filesystem::path destPath(m_configuration_sp->getDestPath());

	if(!boost::filesystem::exists(destPath))
		throw std::runtime_error( "Destination path \"" +
			destPath.string() + "\" not exists");

	if(!boost::filesystem::is_directory(destPath))
		throw std::runtime_error( "Destination path \"" +
			destPath.string() + "\" is not a directory");

    destPath /= origPath.filename();

	if(boost::filesystem::exists(destPath))
		throw std::runtime_error( "Destination path \"" +
			destPath.string() + "\" already exists");

    boost::filesystem::copy(origPath, destPath);
}

}   //namespace
