#include <EventBuffer.h>

#include <boost/thread/locks.hpp>

namespace PreProcessor_ns
{

//==============================================================================
//	EventBuffer::EventBuffer()
//==============================================================================
EventBuffer::EventBuffer(Tango::DeviceImpl* deviceImpl_p) :
	Tango::LogAdapter(deviceImpl_p)
{
		DEBUG_STREAM << "EventBuffer::EventBuffer()" << endl;
}

//==============================================================================
//	EventBuffer::~EventBuffer()
//==============================================================================
EventBuffer::~EventBuffer()
{
	DEBUG_STREAM << "EventBuffer::~EventBuffer()" << endl;
}

//==============================================================================
//	EventBuffer::insertNew()
//==============================================================================
EventBuffer::SP EventBuffer::create(Tango::DeviceImpl* deviceImpl_p)
{
    EventBuffer::SP e_sp(new EventBuffer(deviceImpl_p),
        EventBuffer::Deleter());

    return e_sp;
}

//==============================================================================
//	EventBuffer::insertNew()
//==============================================================================
void EventBuffer::insertNew(boost::filesystem::path path)
{
    DEBUG_STREAM << "EventBuffer::insertNew()" << endl;

	boost::mutex::scoped_lock lock(m_mutex);

    bool inserted = m_buffer.insert(
        std::pair<boost::filesystem::path, EventStatus>(path, UNPROCESSED) ).second;

	if(inserted)
    {
        DEBUG_STREAM << "EventBuffer::insertNew() element "
            << path.string() << " inserted" << endl;

        lock.unlock();

        m_conditionVariable.notify_all();
    }
    else
        WARN_STREAM << "EventBuffer::insertNew() element "
            << path.string() << " duplicated" << endl;
}

//==============================================================================
//	EventBuffer::waitNew()
//==============================================================================
boost::filesystem::path EventBuffer::waitNew()
{
    DEBUG_STREAM << "EventBuffer::waitNew()" << endl;

	boost::mutex::scoped_lock lock(m_mutex);

	do
	{
		std::map<boost::filesystem::path, EventStatus>::iterator it;
		for(it=m_buffer.begin(); it!=m_buffer.end(); it++)
		{
			if(it->second == UNPROCESSED)
			{
                DEBUG_STREAM << "EventBuffer::waitNew() found new element:"
                    << it->first.string() << endl;

				it->second = ASSIGNED;
				return it->first;
			}
		}

        DEBUG_STREAM << "EventBuffer::waitNew() waiting new element" << endl;

		m_conditionVariable.wait(lock);
	}
	while(true);
}

//==============================================================================
//	EventBuffer::markAsProcessed()
//==============================================================================
void EventBuffer::markAsProcessed(boost::filesystem::path path)
{
    DEBUG_STREAM << "EventBuffer::markAsProcessed()" << endl;

    boost::mutex::scoped_lock lock(m_mutex);

    std::map<boost::filesystem::path, EventStatus>::iterator it;

	it = m_buffer.find(path);

	if(it != m_buffer.end())
    {
        switch(it->second)
        {
            case UNPROCESSED:
                ERROR_STREAM << "EventBuffer::markAsProcessed() element "
                    << path.string() << " is marked not processed" << endl;
                break;
            case ASSIGNED:
                it->second = PROCESSED;
                DEBUG_STREAM << "EventBuffer::markAsProcessed() element "
                    << path.string() << " marked as processed" << endl;
                break;
            case PROCESSED:
                ERROR_STREAM << "EventBuffer::markAsProcessed() element "
                    << path.string() << " already marked as processed" << endl;
                break;
        }
    }
    else
        ERROR_STREAM << "EventBuffer::markAsProcessed() element"
            << path.string() << " not found" << endl;
}

//==============================================================================
//	EventBuffer::removeAllProcessed()
//==============================================================================
void EventBuffer::removeAllProcessed()
{
    DEBUG_STREAM << "EventBuffer::removeAllProcessed()" << endl;

    boost::mutex::scoped_lock lock(m_mutex);

    std::map<boost::filesystem::path, EventStatus>::iterator it;
    for(it=m_buffer.begin(); it!=m_buffer.end(); ++it)
    {
        if(it->second == PROCESSED)
        {
            DEBUG_STREAM << "EventBuffer::removeAllProcessed() element "
                << it->first << "will be removed" << endl;

            std::map<boost::filesystem::path, EventStatus>::iterator to_delete = it;

            m_buffer.erase(to_delete);
        }
    }
}

//==============================================================================
//	EventBuffer::size()
//==============================================================================
std::size_t EventBuffer::size()
{
    DEBUG_STREAM << "EventBuffer::size()" << endl;

	boost::mutex::scoped_lock lock(m_mutex);

	return m_buffer.size();
}

}   //namespace
