#include <EventThread.h>
#include <PreProcessor.h>
#include <ScriptManager.h>
#include <WorkerThread.h>

#include <cassert>
#include <unistd.h>
#include <fcntl.h>
#include <cerrno>

#include <boost/filesystem.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/chrono.hpp>

namespace PreProcessor_ns
{

//==============================================================================
//	EventThread::EventThread()
//==============================================================================
EventThread::EventThread(PreProcessor* preProcessor_p,
    Configuration::SP configuration_sp) : Tango::LogAdapter(preProcessor_p),
    m_preProcessor_p(preProcessor_p), m_configuration_sp(configuration_sp)
{
	DEBUG_STREAM << "EventThread::EventThread()" << endl;

    m_state = Tango::OFF;
    m_status = "Event thread not running";
}

//==============================================================================
//	EventThread::~EventThread()
//==============================================================================
EventThread::~EventThread()
{
	DEBUG_STREAM << "EventThread::~EventThread()" << endl;

	if(m_threadGroup_sp)
	{
		m_threadGroup_sp->interrupt_all();

		m_threadGroup_sp->join_all();
	}
}

//==============================================================================
//	EventThread::create()
//==============================================================================
EventThread::SP EventThread::create(PreProcessor* preProcessor_p,
    Configuration::SP configuration_sp)
{
	EventThread::SP e_sp(new EventThread(preProcessor_p, configuration_sp),
        EventThread::Deleter());

	return e_sp;
}

//==============================================================================
//	EventThread::start()
//==============================================================================
void EventThread::start()
{
	DEBUG_STREAM << "EventThread::start()" << endl;

    try
    {
        initEventBuffer();

        initINotify();

        initThreadGroup();
    }
    catch(std::exception& ex)
    {
        std::stringstream error_stream;
        error_stream << "Event thread not running " << ex.what();
        writeState(Tango::ALARM);
        writeStatus(error_stream.str());
    }
    catch(...)
    {
        writeState(Tango::ALARM);
        writeStatus("Event thread unknown exception");
    }
}

//==============================================================================
//	EventThread::stop()
//==============================================================================
void EventThread::stop()
{
	DEBUG_STREAM << "EventThread::stop()" << endl;

	if(m_threadGroup_sp)
	{
		m_threadGroup_sp->interrupt_all();

		m_threadGroup_sp->join_all();
	}

	inotify_rm_watch(m_fileDescriptor, m_watchDescriptor);

	if(m_fileDescriptor) { close(m_fileDescriptor); }
}

//==============================================================================
//	EventThread::readState()
//==============================================================================
Tango::DevState EventThread::readState()
{
    DEBUG_STREAM << "EventThread::readState()" << endl;

    boost::mutex::scoped_lock stateLock(m_stateMutex);

    return m_state;
}

//==============================================================================
//	EventThread::readStatus()
//==============================================================================
std::string EventThread::readStatus()
{
    DEBUG_STREAM << "EventThread::readStatus()" << endl;

    boost::mutex::scoped_lock statusLock(m_statusMutex);

    return m_status;
}

//==============================================================================
//      EventThread::writeState()
//==============================================================================
void EventThread::writeState(Tango::DevState state)
{
    DEBUG_STREAM << "Client::writeState()" << endl;

    boost::mutex::scoped_lock stateLock(m_stateMutex);

    m_state = state;
}

//==============================================================================
//      EventThread::writeStatus()
//==============================================================================
void EventThread::writeStatus(std::string status)
{
    DEBUG_STREAM << "Client::writeStatus()" << endl;

    boost::mutex::scoped_lock statusLock(m_statusMutex);

    m_status = status;
}

//==============================================================================
//	EventThread::initEventBuffer()
//==============================================================================
void EventThread::initEventBuffer() throw(std::runtime_error)
{
    DEBUG_STREAM << "EventThread::initEventBuffer()" << endl;

    m_eventBuffer_sp = EventBuffer::create(m_preProcessor_p);

    std::string watchPath(m_configuration_sp->getWatchPath());

    boost::filesystem::path path(watchPath);

    //Check if watch path exists
    if(!boost::filesystem::exists(path))
        throw std::runtime_error("Watch path \"" +
            watchPath + "\" does not exist");

    //And if it's a directory
    if(!boost::filesystem::is_directory(path))
        throw std::runtime_error("Watch path \"" +
            watchPath + "\" is not a valid directory");
}

//==============================================================================
//	EventThread::initINotify()
//==============================================================================
void EventThread::initINotify() throw(std::runtime_error)
{
    DEBUG_STREAM << "EventThread::initINotify()" << endl;

    if((m_fileDescriptor = inotify_init ()) < 0)
        throw std::runtime_error("INotify initialization error");

	std::string watchPath(m_configuration_sp->getWatchPath());

	uint32_t iNotifyMask = m_configuration_sp->getINotifyMask();

    if((m_watchDescriptor = inotify_add_watch(
        m_fileDescriptor, watchPath.c_str(), iNotifyMask)) < 0)
            throw std::runtime_error("INotify add watch error");

    int flags;
    if((flags = fcntl(m_fileDescriptor,F_GETFL,0)) < 0)
        throw std::runtime_error("File descriptor get flags error");

    if(fcntl(m_fileDescriptor, F_SETFL, flags | O_NONBLOCK) < 0)
        throw std::runtime_error("File descriptor set flags error");
}

//==============================================================================
//	EventThread::initThreadGroup()
//==============================================================================
void EventThread::initThreadGroup() throw(std::runtime_error)
{
    DEBUG_STREAM << "EventThread::initThreadGroup()" << endl;

    ScriptManager::SP fileManager_sp =
        ScriptManager::create(m_preProcessor_p, m_configuration_sp);

    fileManager_sp->checkScriptCompliance();

	WorkerThread worker(m_preProcessor_p, m_eventBuffer_sp,
        fileManager_sp, m_configuration_sp);

	m_threadGroup_sp.reset(new boost::thread_group);

	try
	{
		//Add to thread group event thread
		m_threadGroup_sp->add_thread(
            new boost::thread(&EventThread::eventLoop, this));

        unsigned int workerNumber = m_configuration_sp->getWorkerNumber();

		//Add to thread group worker threads
		for(unsigned int i=0; i<workerNumber; i++)
			m_threadGroup_sp->add_thread(
                new boost::thread(&WorkerThread::workerLoop, worker));
	}
	catch(boost::thread_resource_error& ex)
	{
		std::stringstream error_stream;
		error_stream << "InitThreadGroup: " << ex.what();
		throw std::runtime_error(error_stream.str());
	}
}

//==============================================================================
//	EventThread::eventLoop()
//==============================================================================
void EventThread::eventLoop()
{
	DEBUG_STREAM << "EventThread::eventLoop() starting loop" << endl;

	unsigned int sleepTime = m_configuration_sp->getSleepTime();
	boost::filesystem::path watchPath(m_configuration_sp->getWatchPath());

	while(true)
	{
		try
		{
			char buffer[BUF_LEN];
			int length = 0;

			if((length = read( m_fileDescriptor, buffer, BUF_LEN )) < 0)
			{
				if(errno != EINTR && errno != EAGAIN)
				{
                    writeState(Tango::ALARM);
                    writeStatus("Event thread error on watch path read");
				}
				else
                {
                    writeState(Tango::ON);
                    writeStatus("Event thread running");
                }
			}
			else
            {
                writeState(Tango::ON);
                writeStatus("Event thread new data found");
            }

            struct inotify_event *event;
            for(int i=0; i<length; i += EVENT_SIZE + event->len)
            {
                event = ( struct inotify_event * ) &buffer[ i ];

                //Add path to file name
                boost::filesystem::path file(event->name);
                boost::filesystem::path path(watchPath);
                path /= file;

                //Check if event is a regular file
                if(boost::filesystem::is_regular_file(path))
                    m_eventBuffer_sp->insertNew(path);
            }

            m_eventBuffer_sp->removeAllProcessed();

            DEBUG_STREAM << "EventThread::eventLoop() sleep for " << sleepTime << endl;

			boost::this_thread::sleep_for(boost::chrono::seconds(sleepTime));
		}
		catch(boost::thread_interrupted& ex)
		{
            DEBUG_STREAM << "EventThread::eventLoop() stopping loop" << endl;

            writeState(Tango::OFF);
            writeStatus("Event thread not running");

			break;
		}
        catch(std::exception& ex)
        {
            std::stringstream error_stream;
            error_stream << "Event thread not running " << ex.what();
            writeState(Tango::ALARM);
            writeStatus(error_stream.str());
        }
		catch(...)
		{
            writeState(Tango::ALARM);
            writeStatus("Event thread unknown exception");
		}
	} //thread loop
}

}   //namespace
