#ifndef EVENT_THREAD_H
#define EVENT_THREAD_H

#include <Configuration.h>
#include <EventBuffer.h>

#include <tango.h>

#include <stdexcept>
#include <sys/inotify.h>

#include <boost/shared_ptr.hpp>
#include <boost/thread.hpp>

namespace PreProcessor_ns
{

class PreProcessor;

class EventThread : public Tango::LogAdapter
{
public:
//------------------------------------------------------------------------------
//	[Public] Shared pointer typedef
//------------------------------------------------------------------------------
	typedef boost::shared_ptr<EventThread> SP;

protected:
//------------------------------------------------------------------------------
//	[Protected] Constructor destructor deleter
//------------------------------------------------------------------------------
	EventThread(PreProcessor*, Configuration::SP);

	virtual ~EventThread();

	class Deleter;
	friend class Deleter;
	class Deleter
	{
	public:
		void operator()(EventThread* e) { delete e; }
	};

public:
//------------------------------------------------------------------------------
//	[Public] Class creation method
//------------------------------------------------------------------------------
	static EventThread::SP create(PreProcessor*, Configuration::SP);

//------------------------------------------------------------------------------
//  [Public] Thread management methods
//------------------------------------------------------------------------------
	virtual void start();

	virtual void stop();

//------------------------------------------------------------------------------
//  [Public] Read state and status methods
//------------------------------------------------------------------------------
    virtual Tango::DevState readState();

    virtual std::string readStatus();

protected:
//------------------------------------------------------------------------------
//  [Protected] Write state and status methods
//------------------------------------------------------------------------------
    virtual void writeState(Tango::DevState);

    virtual void writeStatus(std::string);

//------------------------------------------------------------------------------
//	[Protected] Initialization methods
//------------------------------------------------------------------------------
	virtual void initEventBuffer() throw(std::runtime_error);

	virtual void initINotify() throw(std::runtime_error);

	virtual void initThreadGroup() throw(std::runtime_error);

//------------------------------------------------------------------------------
//	[Protected] Event loop method
//------------------------------------------------------------------------------
    virtual void eventLoop();

//------------------------------------------------------------------------------
//	[Protected] Class variables
//------------------------------------------------------------------------------
    //Tango server class pointer
    PreProcessor* m_preProcessor_p;

	//Boost thread group shared pointer
	boost::scoped_ptr<boost::thread_group> m_threadGroup_sp;

    //Configuration shared pointer
    Configuration::SP m_configuration_sp;

	//Event buffer shared pointer
	EventBuffer::SP m_eventBuffer_sp;

    //Tango state synchronization
    boost::mutex m_stateMutex;

    //Tango state variable
    Tango::DevState m_state;

    //Tango status synchronization
    boost::mutex m_statusMutex;

    //Tango status variable
    std::string m_status;

    //INotify file and watch descriptors
    int m_fileDescriptor, m_watchDescriptor;

    //INotify event size
    static const std::size_t EVENT_SIZE = ( sizeof (struct inotify_event) );

    //INotify event buffer length
    static const std::size_t BUF_LEN = ( 1024 * ( EVENT_SIZE + 16 ) );
};

}	//	End of namespace

#endif /*!EVENT_THREAD_H*/
