#include <WorkerThread.h>
#include <PreProcessor.h>

#include <tango.h>

#include <boost/chrono.hpp>

namespace PreProcessor_ns
{

//==============================================================================
//	WorkerThread::WorkerThread()
//==============================================================================
WorkerThread::WorkerThread(PreProcessor* preProcessor_p,
    EventBuffer::SP eventBuffer_sp, ScriptManager::SP fileManager_sp,
    Configuration::SP configuration_sp) : Tango::LogAdapter(preProcessor_p),
    m_preProcessor_p(preProcessor_p), m_eventBuffer_sp(eventBuffer_sp),
    m_fileManager_sp(fileManager_sp), m_configuration_sp(configuration_sp)
{
	DEBUG_STREAM << "WorkerThread::WorkerThread()" << std::endl;
}

//==============================================================================
//	WorkerThread::~WorkerThread()
//==============================================================================
WorkerThread::~WorkerThread()
{
	DEBUG_STREAM << "WorkerThread::~WorkerThread()" << std::endl;
}

//==============================================================================
//	WorkerThread::workerLoop()()
//==============================================================================
void WorkerThread::workerLoop()
{
    DEBUG_STREAM << "WorkerThread::workerLoop(): starting loop" << endl;

    //Sleep time between two verification attempts
    boost::chrono::seconds sleepTime(m_configuration_sp->getSleepTime());

    //Verification attempts timeout
    boost::chrono::steady_clock::duration waitTime =
        boost::chrono::seconds(m_configuration_sp->getWaitTime());

    while(true)
    {
        try
        {
            //Wait new file event to process
            boost::filesystem::path origPath(m_eventBuffer_sp->waitNew());

            std::string fileName = origPath.stem().string();

            //Start timer
            boost::chrono::steady_clock::time_point start =
                boost::chrono::steady_clock::now();

            DEBUG_STREAM << "WorkerThread::workerLoop() processing \""
                << fileName << "\"" << endl;
            try
            {
				//Process only file that match valid regex
				if(m_fileManager_sp->isFileNameValid(origPath))
				{
					ScriptManager::Verified verified;

					//Waiting for the file to be written until timeout
					do
					{
						//Check file state
						verified = m_fileManager_sp->isFileVerified(origPath);

						if(verified == ScriptManager::OK ||
							verified == ScriptManager::FATAL)
							break;

						DEBUG_STREAM << "WorkerThread::workerLoop() waiting \""
							<< fileName << "\" to be written" << endl;

						boost::this_thread::sleep_for(sleepTime);
					}
					while(boost::chrono::steady_clock::now()-start <= waitTime);

					//Exec pre process script
					m_fileManager_sp->preProcessFile(origPath, verified);

					//Copy file to destination
					copyToDestination(origPath, verified);

					//Exec post process script
					m_fileManager_sp->postProcessFile(origPath, verified);

					if(verified == ScriptManager::Verified::OK)
					{
						INFO_STREAM << "WorkerThread::workerLoop() \"" << fileName
							<< "\" is a regular file" << endl;
						m_preProcessor_p->incrementRegularCounter();
					}
					else if(verified == ScriptManager::Verified::WAIT)
					{
						WARN_STREAM << "WorkerThread::workerLoop() \"" << fileName
							<< "\" has reached a wait eof timeout" << endl;
						m_preProcessor_p->incrementWarningCounter();
					}
					else if(verified == ScriptManager::Verified::FATAL)
					{
						WARN_STREAM << "WorkerThread::workerLoop() \"" << fileName
							<< "\" has fatal error" << endl;
						m_preProcessor_p->incrementErrorCounter();
					}
				}
				else
				{
					INFO_STREAM << "WorkerThread::workerLoop() \"" << fileName
						<< "\" is an ignore file" << endl;
					m_preProcessor_p->incrementIgnoredCounter();
				}
            }
            catch(std::exception& ex)
            {
                ERROR_STREAM << "WorkerThread::workerLoop() \"" << fileName
					<< "\" has an error due to " << ex.what() << endl;
                m_preProcessor_p->incrementErrorCounter();
            }

            m_eventBuffer_sp->markAsProcessed(origPath);
        }
		catch(boost::thread_interrupted& ex)
		{
            DEBUG_STREAM << "WorkerThread::workerLoop() interrupt request" << endl;
			break;
		}
		catch(std::exception& ex)
		{
			ERROR_STREAM << "WorkerThread::workerLoop() " << ex.what() << endl;
		}
		catch(...)
		{
			ERROR_STREAM << "WorkerThread::workerLoop() unknown exception" << endl;
		}
    } //thread loop
}

//==============================================================================
//	WorkerThread::moveFile()()
//==============================================================================
void WorkerThread::copyToDestination(boost::filesystem::path& origPath,
	ScriptManager::Verified verified) throw (std::runtime_error)
{
	DEBUG_STREAM << "WorkerThread::copyToDestination()" << endl;

	std::string fileName = origPath.stem().string();

	boost::filesystem::path destPath;

	//Retrieve destination path based verification result
	if(verified == ScriptManager::OK)
	{
		destPath /= m_configuration_sp->getRegularPath();
	}
	else if(verified == ScriptManager::WAIT)
	{
		destPath /= m_configuration_sp->getWarningPath();
	}
	else if(verified == ScriptManager::FATAL)
	{
		destPath /= m_configuration_sp->getErrorPath();
	}

	//If destination is defined then copy file
	if(!destPath.empty())
	{
		if(!boost::filesystem::is_regular_file(origPath))
			throw std::runtime_error( "Origin path \"" +
				origPath.string() + "\" is not a regular file");

		if(!boost::filesystem::exists(destPath))
			throw std::runtime_error( "Destination path \"" +
				destPath.string() + "\" not exists");

		if(!boost::filesystem::is_directory(destPath))
			throw std::runtime_error( "Destination path \"" +
				destPath.string() + "\" is not a directory");

		destPath /= origPath.filename();

		if(boost::filesystem::exists(destPath))
			throw std::runtime_error( "Destination path \"" +
				destPath.string() + "\" already exists");

		boost::filesystem::copy(origPath, destPath);

		DEBUG_STREAM << "WorkerThread::copyToDestination() \"" << fileName
			<< "\" copied to \"" << destPath.parent_path() << "\"" << endl;
	}
	else
	{
		DEBUG_STREAM << "WorkerThread::copyToDestination() \"" << fileName
			<< "\" has not a defined destination" << endl;
	}
}

}   //namespace
