<?php

namespace RAP;

/**
 * RFC 6749 specify that in some situations the client must send an Authorization
 * Basic header containing its credentials (access token in the authorization code
 * flow and refresh token requests).
 */
class ClientAuthChecker {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function validateClientAuth(): void {

        $basic = $this->getBasicAuthArray();

        $clientId = $basic[0];
        $clientSecret = $basic[1];

        $client = $this->locator->getOAuth2ClientDAO()->getOAuth2ClientByClientId($clientId);
        if ($client === null) {
            throw new UnauthorizedException("Client '$clientId' not configured");
        }
        if ($clientSecret !== $client->secret) {
            throw new UnauthorizedException("Invalid client secret");
        }
    }

    public function validateCliClientAuth(): CliClient {

        $basic = $this->getBasicAuthArray();

        $clientId = $basic[0];
        $clientSecret = $basic[1];

        $client = $this->locator->getOAuth2ClientDAO()->getCliClient($clientId, $clientSecret);
        if ($client === null) {
            throw new UnauthorizedException("Client '$clientId' not configured or wrong password");
        }
        return $client;
    }

    private function getBasicAuthArray(): array {
        $headers = apache_request_headers();

        if (!isset($headers['Authorization'])) {
            throw new UnauthorizedException("Missing Authorization header");
        }

        $authorizationHeader = explode(" ", $headers['Authorization']);
        if ($authorizationHeader[0] === "Basic") {
            $basic = explode(':', base64_decode($authorizationHeader[1]));
            if (count($basic) !== 2) {
                throw new BadRequestException("Malformed Basic-Auth header");
            }
            return $basic;
        } else {
            throw new UnauthorizedException("Expected Basic authorization header");
        }
    }

}
