<?php

namespace RAP;

class LoginHandler {

    protected $locator;
    private $identityType;

    public function __construct(Locator $locator, string $identityType) {
        $this->locator = $locator;
        $this->identityType = $identityType;
    }

    public function onIdentityDataReceived(string $typedId, \Closure $fillIdentityData): string {

        $user = $this->locator->getUserDAO()->findUserByIdentity($this->identityType, $typedId);

        if ($user === null) {
            return $this->handleNewIdentity($typedId, $fillIdentityData);
        }

        return $this->getAfterLoginRedirect($user);
    }

    protected function handleNewIdentity(string $typedId, \Closure $fillIdentityData): string {

        $session = $this->locator->getSession();

        if ($session->getUser() !== null && $session->getAction() === 'join') {
            return $this->joinToPreviousUser($session->getUser(), $typedId, $fillIdentityData);
        } else {
            return $this->redirectToTOUCheck($typedId, $fillIdentityData);
        }
    }

    private function joinToPreviousUser(User $user, string $typedId, \Closure $fillIdentityData): string {

        $identity = new Identity($this->identityType);
        $identity->typedId = $typedId;
        $fillIdentityData($identity);

        $user->addIdentity($identity);

        $this->locator->getUserHandler()->saveUser($user);

        $this->locator->getSession()->setUser($user);

        return $this->getAfterLoginRedirect($user);
    }

    /**
     * Stores the data into session and Redirect to Term of Use acceptance page.
     */
    private function redirectToTOUCheck(string $typedId, \Closure $fillIdentityData): string {

        // Create new user
        $user = new \RAP\User();

        $identity = new Identity($this->identityType);
        $identity->typedId = $typedId;
        $fillIdentityData($identity);

        $user->addIdentity($identity);

        $this->locator->getSession()->setUser($user);

        return $this->locator->getBasePath() . '/tou-check';
    }

    public function getAfterLoginRedirect(User $user): string {

        $session = $this->locator->getSession();
        $this->locator->getAuditLogger()->info("LOGIN," . $this->identityType . "," . $user->id);

        if ($session->getOAuth2RequestData() !== null) {
            $session->setUser($user);
            $redirectUrl = $this->locator->getOAuth2RequestHandler()->getRedirectResponseUrl();
            session_destroy();
            return $redirectUrl;
        }

        if ($session->getAction() !== null) {

            $action = $session->getAction();

            if ($action === 'join') {
                if ($session->getUser()->id !== $user->id) {
                    $user = $this->locator->getUserHandler()->joinUsers($session->getUser(), $user);
                }

                // the join is completed
                $action = 'account';
                $session->setAction($action);
            }

            $session->setUser($user);

            if ($action === 'account' || $action === 'admin') {
                return $this->locator->getBasePath() . '/' . $action;
            }
        }

        throw new \Exception("Unable to find a proper redirect");
    }

}
