<?php

namespace RAP;

class MySQLAccessTokenDAO extends BaseMySQLDAO implements AccessTokenDAO {

    public function __construct(Locator $locator) {
        parent::__construct($locator);
    }

    public function createAccessToken(AccessToken $accessToken): AccessToken {

        $dbh = $this->getDBHandler();
        $stmt = $dbh->prepare("INSERT INTO access_token (token, code, user_id, redirect_uri, client_id, scope, creation_time, expiration_time)"
                . " VALUES(:token, :code, :user_id, :redirect_uri, :client_id, :scope, :creation_time, :expiration_time)");

        $scope = null;
        if ($accessToken->scope !== null) {
            $scope = join(' ', $accessToken->scope);
        }

        $params = array(
            ':token' => $accessToken->token,
            ':code' => $accessToken->code,
            ':user_id' => $accessToken->userId,
            ':redirect_uri' => $accessToken->redirectUri,
            ':client_id' => $accessToken->clientId,
            ':scope' => $scope,
            ':creation_time' => $accessToken->creationTime,
            ':expiration_time' => $accessToken->expirationTime
        );

        if ($stmt->execute($params)) {
            return $accessToken;
        } else {
            error_log($stmt->errorInfo()[2]);
            throw new \Exception("SQL error while storing user token");
        }
    }

    public function retrieveAccessTokenFromCode(string $code): ?AccessToken {

        $dbh = $this->getDBHandler();

        // Access token can be retrieved from code in 1 minute from the creation
        $stmt = $dbh->prepare("SELECT token, code, user_id, redirect_uri, client_id, creation_time, expiration_time, scope "
                . " FROM access_token WHERE code = :code AND UNIX_TIMESTAMP() < (creation_time + 60)");
        $stmt->bindParam(':code', $code);

        $stmt->execute();

        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }

        return $this->getAccessTokenFromRow($row);
    }

    public function getAccessToken(string $token): ?AccessToken {

        $dbh = $this->getDBHandler();

        $stmt = $dbh->prepare("SELECT token, code, user_id, redirect_uri, client_id, creation_time, expiration_time, scope "
                . " FROM access_token WHERE token = :token");
        $stmt->bindParam(':token', $token);

        $stmt->execute();

        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }

        return $this->getAccessTokenFromRow($row);
    }

    private function getAccessTokenFromRow(array $row): ?AccessToken {

        $token = new AccessToken();
        $token->token = $row['token'];
        $token->code = $row['code'];
        $token->userId = $row['user_id'];
        $token->redirectUri = $row['redirect_uri'];
        $token->clientId = $row['client_id'];
        $token->creationTime = $row['creation_time'];
        $token->expirationTime = $row['expiration_time'];

        $scope = null;
        if (isset($row['scope'])) {
            $scope = $row['scope'];
        }
        if ($scope !== null && $scope !== '') {
            $token->scope = explode(' ', $scope);
        }

        return $token;
    }

    public function deleteAccessToken($token): void {

        $dbh = $this->getDBHandler();

        $stmt = $dbh->prepare("DELETE FROM access_token WHERE token = :token");
        $stmt->bindParam(':token', $token);
        $stmt->execute();
    }

}
