<?php

use PHPUnit\Framework\TestCase;
use \Firebase\JWT\JWT;

final class IdTokenBuilderTest extends TestCase {

    public function testJWTCreation() {

        $jwksDAOStub = $this->createMock(\RAP\JWKSDAO::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getJWKSDAO')->willReturn($jwksDAOStub);

        $jwksHandler = new \RAP\JWKSHandler($locatorStub);
        $keyPair = $jwksHandler->generateKeyPair();

        $jwksDAOStub->method('getNewestKeyPair')->willReturn($keyPair);

        $user = new \RAP\User();
        $user->id = "user_id";
        $identity = new \RAP\Identity(\RAP\Identity::EDU_GAIN);
        $identity->email = "name@inaf.it";
        $identity->name = "Name";
        $identity->surname = "Surname";
        $identity->primary = true;
        $identity->institution = "INAF";
        $user->addIdentity($identity);

        $daoStub = $this->createMock(\RAP\UserDAO::class);
        $locatorStub->method('getUserDAO')->willReturn($daoStub);
        $daoStub->method('findUserById')->willReturn($user);

        $locatorStub->config = json_decode('{"jwtIssuer": "issuer"}');

        $accessToken = new \RAP\AccessToken();
        $accessToken->token = "ttt";
        $accessToken->scope = ["email", "profile"];
        $accessToken->userId = "user_id";

        $tokenBuilder = new \RAP\IdTokenBuilder($locatorStub);
        $jwt = $tokenBuilder->getIdToken($accessToken);

        $this->assertNotNull($jwt);

        $payload = JWT::decode($jwt, $keyPair->publicKey, [$keyPair->alg]);

        $this->assertEquals("issuer", $payload->iss);
        $this->assertEquals($user->id, $payload->sub);
        $this->assertEquals($user->getCompleteName(), $payload->name);
        $this->assertEquals($identity->name, $payload->given_name);
        $this->assertEquals($identity->surname, $payload->family_name);
        $this->assertEquals($identity->institution, $payload->org);
    }

}
