<?php

use PHPUnit\Framework\TestCase;

final class OAuth2RequestHandlerTest extends TestCase {

    public function testBadRequestExceptionIfMissingClientId(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => null
        ];

        $locatorStub = $this->createMock(\RAP\Locator::class);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testInvalidClientRedirectURI(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "redirect_uri",
            "state" => "state",
            "alg" => null,
            "scope" => "email%20profile"
        ];

        $daoStub = $this->createMock(\RAP\OAuth2ClientDAO::class);
        $daoStub->method('getOAuth2ClientByClientId')->willReturn(new \RAP\OAuth2Client());

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getOAuth2ClientDAO')->willReturn($daoStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testExecuteOAuthStateFlow(): void {

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "redirect_uri",
            "state" => "state",
            "alg" => null,
            "nonce" => null,
            "scope" => "email%20profile"
        ];

        $daoStub = $this->createMock(\RAP\OAuth2ClientDAO::class);
        $client = new \RAP\OAuth2Client();
        $client->redirectUrl = "redirect_uri";
        $daoStub->method('getOAuth2ClientByClientId')->willReturn($client);

        $sessionStub = $this->createMock(\RAP\SessionData::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getOAuth2ClientDAO')->willReturn($daoStub);
        $locatorStub->method('getSession')->willReturn($sessionStub);

        $sessionStub->expects($this->once())
                ->method('setOAuth2Data')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testHandleCheckTokenRequest(): void {

        $accessToken = new \RAP\AccessToken();
        $accessToken->clientId = 'my-client';
        $accessToken->scope = ['openid', 'email'];
        $accessToken->userId = '123';
        $accessToken->expirationTime = time() + 3600;

        $tokenDaoStub = $this->createMock(\RAP\AccessTokenDAO::class);
        $tokenDaoStub->method('getAccessToken')->willReturn($accessToken);

        $user = new \RAP\User();
        $user->id = '123';
        $userDaoStub = $this->createMock(\RAP\UserDAO::class);
        $userDaoStub->method('findUserById')->willReturn($user);

        $idTokenBuilderStub = $this->createMock(\RAP\IdTokenBuilder::class);
        $idTokenBuilderStub->method('getIdToken')->willReturn('id-token');

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getAccessTokenDAO')->willReturn($tokenDaoStub);
        $locatorStub->method('getUserDAO')->willReturn($userDaoStub);
        $locatorStub->method('getIdTokenBuilder')->willReturn($idTokenBuilderStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $result = $requestHandler->handleCheckTokenRequest('abc');

        $this->assertEquals(3600, $result['exp']);
        $this->assertEquals('123', $result['user_name']);
        $this->assertEquals('my-client', $result['client_id']);
        $this->assertEquals('id-token', $result['id_token']);
    }

}
