<?php

namespace RAP;

/**
 * This class is used to build an e-mail message body both in HTML and in its
 * equivalent plaintext.
 */
class MailBodyBuilder {

    private $htmlBody;
    private $textBody;
    private $openedHTMLTags;
    private $editable;

    function __construct() {
        $this->htmlBody = "";
        $this->textBody = "";
        $this->openedHTMLTags = [];
        $this->editable = true;
    }

    private function checkEditable() {
        if (!$this->editable) {
            throw new \Exception("You cannot edit the body after it has been generated");
        }
    }

    public function addText($text) {
        $this->checkEditable();
        $this->htmlBody .= $text;
        $this->textBody .= $text;
        return $this;
    }

    public function addLineBreak() {
        $this->checkEditable();
        $this->htmlBody .= "<br>\n";
        $this->textBody .= "\n";
        return $this;
    }

    public function addHr() {
        $this->checkEditable();
        $this->htmlBody .= "<hr/>\n";
        $this->textBody .= "\n---------------------\n";
        return $this;
    }

    public function addLinkWithDescription($url, $text) {
        $this->checkEditable();
        $this->htmlBody .= '<a href="' . $url . '" target="blank_">' . $text . "</a>\n";
        $this->textBody .= $text . " ( " . $url . " ) ";
        return $this;
    }

    public function addLink($url) {
        $this->checkEditable();
        $this->htmlBody .= '<a href="' . $url . '" target="blank_">' . $url . "</a>\n";
        $this->textBody .= $url . " ";
        return $this;
    }

    public function addEmailAddress($email, $description) {
        $this->checkEditable();
        $this->htmlBody .= '<a href="mailto:' . $email . '">' . $description . '</a>';
        $this->textBody .= $description . " (" . $email . ")";
        return $this;
    }

    private function openHTMLTag($tag, $equivalentPlainText) {
        $this->checkEditable();
        if (in_array($tag, $this->openedHTMLTags)) {
            throw new \Exception("You are already inside a " . $tag . " tag!");
        }
        $this->openedHTMLTags[] = $tag;
        $this->htmlBody .= "<" . $tag . ">";
        $this->textBody .= $equivalentPlainText;
        return $this;
    }

    private function closeHTMLTag($tag, $equivalentPlainText) {
        $this->checkEditable();
        if ($this->openedHTMLTags[count($this->openedHTMLTags) - 1] !== $tag) {
            throw new \Exception("You are not inside a " . $tag . " tag!");
        }
        array_pop($this->openedHTMLTags);
        $this->htmlBody .= "</" . $tag . ">";
        $this->textBody .= $equivalentPlainText;
        return $this;
    }

    public function startBold() {
        return $this->openHTMLTag("strong", "*");
    }

    public function endBold() {
        return $this->closeHTMLTag("strong", "*");
    }

    public function startParagraph() {
        return $this->openHTMLTag("p", "");
    }

    public function endParagraph() {
        return $this->closeHTMLTag("p", "\n");
    }

    public function startList() {
        return $this->openHTMLTag("ul", "\n");
    }

    public function startListItem() {
        return $this->openHTMLTag("li", " * ");
    }

    public function endListItem() {
        return $this->closeHTMLTag("li", "\n");
    }

    public function endList() {
        return $this->closeHTMLTag("ul", "\n");
    }

    private function checkEnd() {
        if (count($this->openedHTMLTags) > 0) {
            $unclosedTags = "";
            foreach ($this->openedHTMLTags as $tag) {
                $unclosedTags .= $tag . " ";
            }
            throw new \Exception("You must close all tags before generating email body! Unclosed tags: " . $unclosedTags);
        }
    }

    private function setCompatibleLineBreaks($value) {
        return str_replace("\n", "\n\r", $value);
    }

    private function finalizeBodyIfNecessary() {
        if ($this->editable) {
            $this->checkEnd();

            $this->htmlBody = $this->setCompatibleLineBreaks($this->htmlBody);
            $this->textBody = $this->setCompatibleLineBreaks($this->textBody);

            $this->editable = false;
        }
    }

    public function getTextPlainBody() {
        $this->finalizeBodyIfNecessary();
        return $this->textBody;
    }

    public function getHTMLBody() {
        $this->finalizeBodyIfNecessary();
        return $this->htmlBody;
    }

}
