<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

use \PHPMailer\PHPMailer\PHPMailer;

/**
 * Manage mail sending.
 * Currently used only for join email messages.
 */
class MailSender {

    private $serverName;
    private $basePath;
    private $mbb;

    public function __construct($serverName, $basePath) {
        $this->serverName = $serverName;
        $this->basePath = $basePath;
        $this->mbb = new MailBodyBuilder();
    }

    private function addDescriptionItem($key, $value) {
        $this->mbb->startBold()
                ->addText($key)
                ->endBold()
                ->addText(": " . $value)
                ->addLineBreak();
    }

    /**
     * Send the email for confirming the join request.
     * @param \RAP\User $recipientUser user target of the join requests: he/she
     * will receive the email containing the confirmation link
     * @param \RAP\User $applicantUser user that have requested the join
     * @param string $token the join token
     */
    public function sendJoinEmail(User $recipientUser, User $applicantUser, $token) {

        global $auditLog;

        $confirmJoinURL = $this->basePath . '/confirm-join?token=' . $token;

        $this->mbb->startParagraph()
                ->addText("Dear IA2 user,")
                ->addLineBreak()
                ->addText("the following user requested to join your accounts on the ")
                ->addLinkWithDescription("https://sso.ia2.inaf.it/rap-ia2/", "RAP facility")
                ->addText(":")
                ->endParagraph();

        foreach ($applicantUser->identities as $identity) {

            $this->addDescriptionItem("Type", $identity->type);
            if ($identity->name !== null) {
                $this->addDescriptionItem("Name", $identity->name);
            }
            if ($identity->surname !== null) {
                $this->addDescriptionItem("Surname", $identity->surname);
            }
            $this->addDescriptionItem("E-mail", $identity->email);
            if ($identity->eppn !== null) {
                $this->addDescriptionItem("Eppn", $identity->eppn);
            }
            if ($identity->institution !== null) {
                $this->addDescriptionItem("Institution", $identity->institution);
            }

            $this->mbb->addLineBreak();
        }

        $this->mbb->startParagraph()
                ->addText("If you and this user are ")
                ->startBold()
                ->addText("the same person")
                ->endBold()
                ->addText(" click on the following link for joining your accounts: ")
                ->addLink($confirmJoinURL)
                ->addLineBreak()
                ->addText("Otherwise you can ignore this email.")
                ->endParagraph()
                //
                ->startParagraph()
                ->startBold()
                ->addText("Please don't use this functionality for sharing resources between your coworkers")
                ->endBold()
                ->addText(", use ")
                ->addLinkWithDescription("https://sso.ia2.inaf.it/grouper", "Grouper")
                ->addText(" for that.")
                ->endParagraph()
                //
                ->addLineBreak()
                ->startBold()
                ->addText("*** This is an automatically generated email, please do not reply to this message ***")
                ->endBold()
                ->addLineBreak()
                ->addText("If you need information please contact ")
                ->addEmailAddress("ia2@oats.inaf.it", "IA2 Staff");

        $mail = new PHPMailer(true); // Passing `true` enables exceptions
        try {

            $toAddress = $recipientUser->getPrimaryEmail();

            $mail->isSMTP();
            $mail->Port = 25;
            $mail->setFrom("noreply@" . $this->serverName, 'IA2 SSO');
            $mail->addAddress($toAddress);
            $mail->CharSet = 'utf-8';
            $mail->Subject = "IA2 RAP: Join request";
            $mail->Body = $this->mbb->getHTMLBody();
            $mail->AltBody = $this->mbb->getTextPlainBody();

            $auditLog->info("JOIN email. Sending to " . $toAddress);
            $mail->send();
        } catch (\Exception $ex) {
            error_log($ex->getMessage());
            throw $ex;
        }
    }

}
