<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Data model representing an item of the result of an user search. This is used
 * in order to display the dropdown menu for user selection in the join modal
 * dialog avoiding exposing the user identifiers in the AJAX calls.
 * This data is stored into a list inside the SessionData object. The user will
 * select one of the item by index and not by the user ID.
 */
class UserSearchResult {

    // The user object is wrapped by this class and hidden to AJAX.
    private $user;
    // Only this text is returned to the AJAX call.
    // See gui-backend.php (/user?search endpoint)
    public $userDisplayText;

    public static function buildFromUser(User $user) {
        $usr = new UserSearchResult();
        $usr->user = $user;

        $nameAndSurname = null;
        $email = null;
        $identityTypes = [];
        foreach ($user->identities as $identity) {
            array_push($identityTypes, $identity->getUIType());
            if ($nameAndSurname === null && $identity->name !== null && $identity->surname !== null) {
                $nameAndSurname = $identity->name . ' ' . $identity->surname;
            }
            if ($email === null) {
                $email = $identity->email;
            }
        }

        // Building display text string        
        $displayText = "";

        if ($nameAndSurname !== null) {
            $displayText .= $nameAndSurname;
        } else {
            $displayText .= $email;
        }

        $displayText .= ' (';
        $firstIdentity = true;
        foreach ($identityTypes as $type) {
            if (!$firstIdentity) {
                $displayText .= '+';
            }
            $displayText .= $type;
            $firstIdentity = false;
        }
        $displayText .= ')';

        $usr->userDisplayText = $displayText;

        return $usr;
    }

    public function getUser() {
        return $this->user;
    }

}
