<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * This class wraps various objects that need to be stored into the session in
 * order to provide an object oriented transparent session management.
 */
class SessionData {

    private $dao;
    private $callbackURL;
    private $callbackTitle;
    private $callbackLogo;
    private $callbackAuth;
    public $user;
    public $userSearchResults;
    public $x509DataToRegister;
    // user which is going to perform the login (we need to store this in the
    // session because we need to check the Terms of Use user consensus, so we
    // redirect to another page after retrieving user data.
    public $userToLogin;

    /**
     * @todo: move DAO away from here
     */
    public function __construct(DAO $dao) {
        $this->dao = $dao;
    }

    /**
     * Store the data into the $_SESSION PHP variable
     */
    public function save() {
        $_SESSION['SessionData'] = $this;
    }

    /**
     * Retrieve the SessionData object from the $_SESSION PHP variable. Create a
     * new one if it is necessary.
     * @param \RAP\DAO $dao
     * @return \RAP\SessionData the SessionData object
     */
    public static function get(DAO $dao) {

        if (!isset($_SESSION['SessionData'])) {
            $session = new SessionData($dao);
            $session->save();
        }
        return $_SESSION['SessionData'];
    }

    public function setCallbackURL(CallbackHandler $callbackHandler, $callbackURL) {
        $this->callbackURL = $callbackHandler->filterCallbackURL($callbackURL);
        $this->callbackTitle = $callbackHandler->getCallbackTitle($callbackURL);
        $this->callbackLogo = $callbackHandler->getCallbackLogo($callbackURL);
        $this->callbackAuth = $callbackHandler->getCallbackAuth($callbackURL);
        $this->save();
    }

    public function getCallbackURL() {
        return $this->callbackURL;
    }

    public function getCallbackTitle() {
        return $this->callbackTitle;
    }

    public function getCallbackLogo() {
        return $this->callbackLogo;
    }

    public function getCallbackAuth() {
        return $this->callbackAuth;
    }

    /**
     * Perform a user search and store the results inside the session. This is
     * used for achieving the user selection using the dropdown menu in the join
     * request modal.
     * @param string $searchText
     */
    public function searchUser($searchText) {
        $users = $this->dao->searchUser($searchText);

        $this->userSearchResults = [];
        foreach ($users as $user) {
            // this search shouldn't contains the user itself
            if ($user->id !== $this->user->id) {
                $searchResult = UserSearchResult::buildFromUser($user);
                array_push($this->userSearchResults, $searchResult);
            }
        }

        $this->save();
    }

    /**
     * Update the user data model stored into the session after the primary
     * identity has changed, in order to avoid reading again the user data from
     * the database.
     * @param int $identityId
     */
    public function updatePrimaryIdentity($identityId) {
        foreach ($this->user->identities as $identity) {
            $identity->primary = ($identity->id === $identityId);
        }
    }

}
