<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Perform operations on users.
 */
class UserHandler {

    private $dao;

    public function __construct(DAO $dao) {
        $this->dao = $dao;
    }

    /**
     * Update user information into the database, creating a new user or adding
     * new identities to it.
     * @param \RAP\User $user
     */
    public function saveUser(User $user) {

        $primarySpecified = true;

        // If new user
        if ($user->id === null) {
            $primarySpecified = false;
            $user->id = $this->dao->createUser();
        }

        foreach ($user->identities as $identity) {
            if ($identity->id === null) {
                $identity->id = $this->dao->insertIdentity($identity, $user->id);
                if (!$primarySpecified) {
                    $this->dao->setPrimaryIdentity($user->id, $identity->id);
                    $identity->primary = true;
                }
            }
        }
    }

    public function findUserByIdentity($type, $identifier) {

        return $this->dao->findUserByIdentity($type, $identifier);
    }

    /**
     * Build an URL for the web service endpoint that needs to be called in order
     * to move groups from one user to the other during a join operation.
     * @return string grouper URL for the PrepareToJoinServlet
     */
    private function getJoinURL() {
        $joinURL = $this->grouperConfig['wsURL'];

        if (substr($joinURL, -1) !== '/') {
            $joinURL .= '/';
        }
        $joinURL .= 'ia2join';

        return $joinURL;
    }

    public function joinUsers($userId1, $userId2) {

        // Call Grouper for moving groups and privileges from one user to the other
        if ($this->grouperConfig !== null) {

            //create cURL connection
            $conn = curl_init($this->getJoinURL());

            //set options
            curl_setopt($conn, CURLOPT_CONNECTTIMEOUT, 30);
            curl_setopt($conn, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($conn, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($conn, CURLOPT_FOLLOWLOCATION, 1);
            curl_setopt($conn, CURLOPT_USERPWD, $this->grouperConfig['user'] . ":" . $this->grouperConfig['password']);

            //set data to be posted
            curl_setopt($conn, CURLOPT_POST, 1);
            curl_setopt($conn, CURLOPT_POSTFIELDS, "subject1Id=RAP:$userId1&subject2Id=RAP:$userId2");

            //perform the request
            $response = curl_exec($conn);
            $info = curl_getinfo($conn);

            if ($info['http_code'] === 200) {
                curl_close($conn);
            } else {
                //show information regarding the error
                curl_close($conn);
                http_response_code(500);
                die('Error: Grouper response code: ' . $info['http_code']);
            }
        }

        // Call DAO for performing join operation into the RAP database.
        $this->dao->joinUsers($userId1, $userId2);
    }

}
