<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Manage callback URL validation and redirection
 */
class CallbackHandler {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    /**
     * If a callback URL is not in the configured list we should return null.
     */
    public function filterCallbackURL($callbackURL) {
        foreach ($this->callbacks as $callback) {
            if ($callback['url'] === $callbackURL) {
                return $callbackURL;
            }
        }
        return null;
    }

    /**
     * Each callback has a title and a logo in order to avoid confusion in users
     * and show in which application they are logging in using RAP.
     * @param type $callbackURL
     * @return type the callback title or null if the callback URL is not listed
     * in configuration file or it doesn't have a title.
     */
    public function getCallbackTitle($callbackURL) {

        foreach ($this->callbacks as $callback) {
            if ($callback['url'] === $callbackURL) {
                return $callback['title'];
            }
        }

        return null;
    }

    /**
     * Each callback has a title and a logo in order to avoid confusion in users
     * and show in which application they are logging in using RAP.
     * @param type $callbackURL
     * @return type the callback logo or null if the callback URL is not listed
     * in configuration file or it doesn't have a logo.
     */
    public function getCallbackLogo($callbackURL) {

        foreach ($this->callbacks as $callback) {
            if ($callback['url'] === $callbackURL) {
                if (array_key_exists('logo', $callback)) {
                    return $callback['logo'];
                } else {
                    return null;
                }
            }
        }

        return null;
    }

    public function manageLoginRedirect(User $user, SessionData $session) {
        
        if($session->getOAuth2Data() !== null) {
            $session->user = $user;
            $session->save();
            $redirectUrl = $this->locator->getOAuth2RequestHandler()->getCodeResponseUrl();
            $session->setOAuth2Data(null);
            header('Location: ' . $redirectUrl);
            die();
        }
        
        if ($session->getCallbackURL() === null) {
            http_response_code(401);
            die("Unauthorized callback URL");
        }

        if ($session->getCallbackURL() === $this->basePath . '/') {
            // Login in session
            $session->user = $user;
            $session->save();
            // Return to index
            header('Location: ' . $this->basePath);
            die();
        } else {
            // External login using token
            header('Location: ' . $this->getLoginWithTokenURL($user->id, $session->getCallbackURL()));
            die();
        }
    }

    public function getLoginWithTokenURL($userId, $callbackURL) {
        $token = Util::createNewToken();
        $this->dao->createLoginToken($token, $userId);
        return $callbackURL . '?token=' . $token;
    }

}
