<?php

namespace RAP;

class OAuth2RequestHandler {

    private $locator;

    public function __construct(\RAP\Locator $locator) {
        $this->locator = $locator;
    }

    public function handleAuthorizeRequest() {

        if (!isset($_REQUEST['client_id'])) {
            throw new BadRequestException("Client id is required");
        }

        if (!isset($_REQUEST['redirect_uri'])) {
            throw new BadRequestException("Redirect URI is required");
        }

        $clientId = $_REQUEST['client_id'];
        $redirectUrl = $_REQUEST['redirect_uri'];

        $client = $this->locator->getDAO()->getOAuth2ClientByClientId($clientId);
        if ($client === null) {
            throw new BadRequestException("Invalid client id: " . $clientId);
        }
        if ($client->redirectUrl !== $redirectUrl) {
            throw new BadRequestException("Invalid client redirect URI: " . $redirectUrl);
        }

        $alg;
        if (isset($_REQUEST['alg'])) {
            $alg = $_REQUEST['alg'];
        } else {
            $alg = "RS256";
        }

        if (isset($_GET['code'])) {
            
        } else {
            $this->executeStateFlow($client);
        }
    }

    private function executeStateFlow(OAuth2Client $client) {

        if (!isset($_REQUEST['state'])) {
            throw new BadRequestException("State is required");
        }

        // Storing OAuth2 data in session
        $oauth2Data = new \RAP\OAuth2Data();
        $oauth2Data->clientId = $client->client;
        $oauth2Data->redirectUrl = $client->redirectUrl;
        $oauth2Data->state = $_REQUEST['state'];

        $session = $this->locator->getSession();
        $session->setOAuth2Data($oauth2Data);
    }

    public function getCodeResponseUrl(): string {

        $session = $this->locator->getSession();

        $accessToken = new \RAP\AccessToken();
        $accessToken->code = base64_encode(bin2hex(openssl_random_pseudo_bytes(64)));
        $accessToken->token = base64_encode(bin2hex(openssl_random_pseudo_bytes(128)));
        $accessToken->userId = $session->user->id;
        $accessToken->clientId = $session->getOAuth2Data()->clientId;
        $accessToken->redirectUri = $session->getOAuth2Data()->redirectUrl;
        //$accessToken->scope = 

        $this->locator->getAccessTokenDAO()->createAccessToken($accessToken);

        $state = $session->getOAuth2Data()->state;

        $redirectUrl = $session->getOAuth2Data()->redirectUrl
                . '?code=' . $accessToken->code . '&scope=profile&state=' . $state;

        return $redirectUrl;
    }

    public function handleAccessTokenRequest(): array {

        $this->validateAccessTokenRequest();

        $code = filter_input(INPUT_POST, 'code', FILTER_SANITIZE_STRING);
        $accessToken = $this->locator->getAccessTokenDAO()->retrieveAccessTokenFromCode($code);
        
        if($accessToken === null) {
            throw new BadRequestException("No token for given code");
        }
        
        $this->validateParametersMatching();

        $token = [];
        $token['access_token'] = $accessToken->token;
        $token['token_type'] = 'bearer';
        $token['expires_in'] = 300;
        error_log($accessToken->creationTime);
        error_log($accessToken->expirationTime);

        if ($accessToken->scope !== null) {
            $token['id_token'] = $this->locator->getIdTokenBuilder()->getIdToken($accessToken->userId, 'RS256');
        }

        return $token;
    }

    private function validateAccessTokenRequest() {

        if (!isset($_POST['grant_type'])) {
            throw new BadRequestException("Client id is required");
        } else if ($_POST['grant_type'] !== 'authorization_code') {
            throw new BadRequestException("grant_type must be authorization_code");
        }

        if (!isset($_POST['code'])) {
            throw new BadRequestException("Client id is required");
        }

        if (!isset($_POST['redirect_uri'])) {
            throw new BadRequestException("Redirect URI is required");
        }

        // Note: theorically the standard wants also the client_id here,
        // however some clients don't send it
    }

    private function validateParametersMatching() {
        
    }

    public function handleCheckTokenRequest(): array {

        if (!isset($_POST['token'])) {
            throw new BadRequestException("Access token id is required");
        }

        $accessToken = filter_input(INPUT_POST, 'token', FILTER_SANITIZE_STRING);

        //if($accessToken)

        $result = [];
        $result['exp'] = 3600;
        $result['user_name'] = "test";
        $result['client_id'] = "gms";
        $result['scope'] = "profile";

        return $result;
    }

}
