<?php

/**
 * Front Controller using http://flightphp.com/
 * In all these calls user session must exist, so we have to start it at the 
 * beginning using the session_start() function.
 */
//

function setCallback($callback) {
    global $session, $callbackHandler;

    if (!isset($callback) || $callback === '') {
        $callback = null;
    }
    $session->setCallbackURL($callbackHandler, $callback);
    return $session->getCallbackURL();
}

/**
 * Display the main page (authentication method selection) or the available
 * services list if a valid callback is not found
 */
Flight::route('/', function() {

    session_start();
    global $locator;

    $action = Flight::request()->query['action'];

    switch ($action) {
        case "oaut2client":
            $clientId = $locator->getSession()->getOAuth2Data()->clientId;
            $client = $locator->getDAO()->getOAuth2ClientByClientId($clientId);
            $authPageModel = new \RAP\AuthPageModel($locator, $client);
            renderMainPage($authPageModel);
            break;
        default:
            session_destroy();
            Flight::render('services-list.php', array('title' => 'RAP',
                'version' => $locator->getVersion(),
                'action' => $locator->getBasePath() . '/'));
            break;
    }
});

function renderMainPage(RAP\AuthPageModel $authPageModel) {
    global $locator;
    Flight::render('main-page.php', array('title' => 'RAP',
        'version' => $locator->getVersion(), 'model' => $authPageModel));
}

Flight::route('GET /auth/oauth2/authorize', function() {

    session_start();
    global $locator;

    $requestHandler = new \RAP\OAuth2RequestHandler($locator);
    $requestHandler->handleAuthorizeRequest();

    Flight::redirect('/?action=oaut2client');
});

Flight::route('POST /auth/oauth2/token', function() {

    global $locator;

    $requestHandler = new \RAP\OAuth2RequestHandler($locator);
    $token = $requestHandler->handleAccessTokenRequest();

    Flight::json($token);
});

Flight::route('POST /auth/oauth2/check_token', function() {
    
    global $locator;

    $requestHandler = new \RAP\OAuth2RequestHandler($locator);
    $token = $requestHandler->handleCheckTokenRequest();

    Flight::json($token);
});

Flight::route('GET /auth/oidc/jwks', function() {
    
    global $locator;

    $jwksHandler = new \RAP\JWKSHandler($locator);
    $jwks = $jwksHandler->getJWKS();

    Flight::json($jwks);
});

Flight::route('GET /logout', function() {
    session_start();
    session_destroy();
    Flight::redirect('/');
});

function sendAuthRedirect($url) {
    startSession();
    // reload callback from query to avoid problem with session shared between 
    // multiple browser tabs
    setCallback(Flight::request()->query['callback']);
    Flight::redirect($url);
}

Flight::route('/auth/social/google', function() {
    session_start();
    global $locator;
    $googleLogin = new \RAP\GoogleLogin($locator);
    $googleLogin->call();
});

Flight::route('/facebook', function() {
    sendAuthRedirect('/auth/social/facebook_login.php');
});

Flight::route('/linkedIn', function() {
    sendAuthRedirect('/auth/social/linkedin_login.php');
});

Flight::route('/eduGAIN', function() {
    sendAuthRedirect('/auth/saml2/aai.php');
});

Flight::route('/x509', function() {
    sendAuthRedirect('/auth/x509/certlogin.php');
});

Flight::route('/local', function() {
    global $AUTHENTICATION_METHODS;
    sendAuthRedirect($AUTHENTICATION_METHODS['DirectIdP']['url']);
});

/**
 * Render the page for selecting the correct name and username from candidates
 * list during a X.509 registration.
 */
Flight::route('GET /x509-name-surname', function() {

    startSession();
    global $session, $BASE_PATH, $VERSION;

    if ($session->x509DataToRegister !== null && $session->x509DataToRegister->name === null) {
        Flight::render('x509-name-surname.php', array('title' => 'Select name and surname',
            'version' => $VERSION,
            'fullName' => $session->x509DataToRegister->fullName,
            'candidateNames' => $session->x509DataToRegister->candidateNames));
    } else {
        // Redirect to index
        header("Location: " . $BASE_PATH);
        die();
    }
});

/**
 * Complete the X.509 registration selecting the correct name and surname specified
 * by the user.
 */
Flight::route('POST /submit-x509-name', function() {

    $selectedNameIndex = Flight::request()->data['selected-name'];

    error_log('index=' . $selectedNameIndex);

    startSession();
    global $session, $BASE_PATH;

    if ($session->x509DataToRegister !== null) {
        $session->x509DataToRegister->selectCandidateName($selectedNameIndex);
        $session->userToLogin = $session->x509DataToRegister->toUser();
        $session->x509DataToRegister = null;
        $session->save();
        header("Location: " . $BASE_PATH . '/tou-check');
        die();
    } else {
        die('X.509 data not returned');
    }
});

Flight::route('GET /tou-check', function() {

    startSession();
    global $session, $BASE_PATH, $VERSION;

    if ($session->userToLogin === null) {
        die("User data not retrieved.");
    } else {
        Flight::render('tou-check.php', array('title' => 'Terms of Use acceptance',
            'user' => $session->userToLogin,
            'version' => $VERSION,
            'registration_url' => $BASE_PATH . '/register'));
    }
});

Flight::route('GET /register', function() {

    startSession();
    global $session, $userHandler, $auditLog, $callbackHandler;

    if ($session->userToLogin === null) {
        die("User data not retrieved.");
    } else {

        $user = $session->userToLogin;
        $userHandler->saveUser($user);

        $session->userToLogin = null;
        $session->save();

        $auditLog->info("LOGIN," . $user->identities[0]->type . "," . $user->id);
        $callbackHandler->manageLoginRedirect($user, $session);
    }
});

include 'admin.php';
