<?php

namespace RAP;

class MySQLOAuth2ClientDAO extends BaseMySQLDAO implements OAuth2ClientDAO {

    public function __construct($config) {
        parent::__construct($config);
    }

    public function getOAuth2Clients(): array {

        $dbh = $this->getDBHandler();

        $clientsMap = $this->getClientsMap($dbh);
        $this->loadAuthenticationMethods($dbh, $clientsMap);
        $this->loadScopeAudienceMapping($dbh, $clientsMap);

        $clients = [];
        foreach ($clientsMap as $id => $client) {
            array_push($clients, $client);
        }

        return $clients;
    }

    private function getClientsMap(PDO $dbh): array {

        // Load clients info
        $queryClient = "SELECT id, title, icon, client, secret, redirect_url, scope, home_page, show_in_home FROM oauth2_client";
        $stmtClients = $dbh->prepare($queryClient);
        $stmtClients->execute();

        $clientsMap = [];

        foreach ($stmtClients->fetchAll() as $row) {
            $client = new OAuth2Client();
            $client->id = $row['id'];
            $client->title = $row['title'];
            $client->icon = $row['icon'];
            $client->client = $row['client'];
            $client->secret = $row['secret'];
            $client->redirectUrl = $row['redirect_url'];
            $client->scope = $row['scope'];
            $client->homePage = $row['home_page'];
            $client->showInHome = boolval($row['show_in_home']);
            $clientsMap[$client->id] = $client;
        }

        return $clientsMap;
    }

    private function loadAuthenticationMethods(PDO $dbh, array $clientsMap): void {

        $queryAuthNMethods = "SELECT client_id, auth_method FROM oauth2_client_auth_methods";

        $stmtAuthNMethods = $dbh->prepare($queryAuthNMethods);
        $stmtAuthNMethods->execute();

        foreach ($stmtAuthNMethods->fetchAll() as $row) {
            $id = $row['client_id'];
            array_push($clientsMap[$id]->authMethods, $row['auth_method']);
        }
    }

    private function loadScopeAudienceMapping(PDO $dbh, array $clientsMap): void {

        $query = "SELECT client_id, scope, audience FROM oauth2_client_scope_audience_mapping";

        $stmt = $dbh->prepare($query);

        foreach ($stmt->fetchAll() as $row) {
            $id = $row['client_id'];

            if (array_key_exists($id, $clientsMap)) {
                $client = $clientsMap[$id];
                $client->scopeAudienceMap[$row['scope']] = $row['audience'];
            }

            array_push($clientsMap[$id]->authMethods, $row['auth_method']);
        }
    }

    function createOAuth2Client(OAuth2Client $client): OAuth2Client {
        $dbh = $this->getDBHandler();

        try {
            $dbh->beginTransaction();

            $stmt = $dbh->prepare("INSERT INTO `oauth2_client`(`title`, `icon`, `client`, `secret`, `redirect_url`, `scope`, home_page, show_in_home)"
                    . " VALUES(:title, :icon, :client, :secret, :redirect_url, :scope, :home_page, :show_in_home)");

            $stmt->bindParam(':title', $client->title);
            $stmt->bindParam(':icon', $client->icon);
            $stmt->bindParam(':client', $client->client);
            $stmt->bindParam(':secret', $client->secret);
            $stmt->bindParam(':redirect_url', $client->redirectUrl);
            $stmt->bindParam(':scope', $client->scope);
            $stmt->bindParam(':home_page', $client->homePage);
            $stmt->bindParam(':show_in_home', $client->showInHome, \PDO::PARAM_INT);

            $stmt->execute();

            $client->id = $dbh->lastInsertId();

            foreach ($client->authMethods as $method) {
                $stmt = $dbh->prepare("INSERT INTO `oauth2_client_auth_methods`(`client_id`, `auth_method`)"
                        . " VALUES(:client_id, :auth_method)");

                $stmt->bindParam(':client_id', $client->id);
                $stmt->bindParam(':auth_method', $method);

                $stmt->execute();
            }

            $dbh->commit();
        } catch (Exception $ex) {
            $dbh->rollBack();
            throw $ex;
        }

        return $client;
    }

    function updateOAuth2Client(OAuth2Client $client): OAuth2Client {
        $dbh = $this->getDBHandler();

        try {
            $dbh->beginTransaction();

            $stmt = $dbh->prepare("UPDATE `oauth2_client` SET `title` = :title, `icon` = :icon, "
                    . " `client` = :client, `secret` = :secret, `redirect_url` = :redirect_url, `scope` = :scope, "
                    . " `home_page` = :home_page, `show_in_home` = :show_in_home"
                    . " WHERE id = :id");

            $stmt->bindParam(':title', $client->title);
            $stmt->bindParam(':icon', $client->icon);
            $stmt->bindParam(':client', $client->client);
            $stmt->bindParam(':secret', $client->secret);
            $stmt->bindParam(':redirect_url', $client->redirectUrl);
            $stmt->bindParam(':scope', $client->scope);
            $stmt->bindParam(':home_page', $client->homePage);
            $stmt->bindParam(':show_in_home', $client->showInHome, \PDO::PARAM_INT);
            $stmt->bindParam(':id', $client->id);

            $stmt->execute();

            // Delete old authentication methods
            $stmt = $dbh->prepare("DELETE FROM oauth2_client_auth_methods WHERE client_id = :id");
            $stmt->bindParam(':id', $client->id);

            $stmt->execute();

            // Re-add authentication methods
            foreach ($client->authMethods as $method) {
                $stmt = $dbh->prepare("INSERT INTO `oauth2_client_auth_methods`(`client_id`, `auth_method`)"
                        . " VALUES(:client_id, :auth_method)");

                $stmt->bindParam(':client_id', $client->id);
                $stmt->bindParam(':auth_method', $method);

                $stmt->execute();
            }

            $dbh->commit();
        } catch (Exception $ex) {
            $dbh->rollBack();
            throw $ex;
        }

        return $client;
    }

    function deleteOAuth2Client($clientId) {
        $dbh = $this->getDBHandler();
        try {
            $dbh->beginTransaction();

            $stmt = $dbh->prepare("DELETE FROM `oauth2_client_auth_methods` WHERE client_id = :id");
            $stmt->bindParam(':id', $clientId);
            $stmt->execute();

            $stmt = $dbh->prepare("DELETE FROM `oauth2_client` WHERE id = :id");
            $stmt->bindParam(':id', $clientId);
            $stmt->execute();

            $dbh->commit();
        } catch (Exception $ex) {
            $dbh->rollBack();
            throw $ex;
        }
    }

    function getOAuth2ClientByClientId($clientId): ?OAuth2Client {
        $dbh = $this->getDBHandler();

        // Load clients info
        $queryClient = "SELECT id, title, icon, client, secret, redirect_url, scope, home_page, show_in_home FROM oauth2_client WHERE client = :client";
        $stmtClient = $dbh->prepare($queryClient);
        $stmtClient->bindParam(':client', $clientId);
        $stmtClient->execute();

        $result = $stmtClient->fetchAll();

        if (count($result) === 0) {
            return null;
        }
        if (count($result) > 1) {
            throw new Exception("Found multiple clients associated to the same client id!");
        }

        $row = $result[0];

        $client = new OAuth2Client();
        $client->id = $row['id'];
        $client->title = $row['title'];
        $client->icon = $row['icon'];
        $client->client = $row['client'];
        $client->secret = $row['secret'];
        $client->redirectUrl = $row['redirect_url'];
        $client->scope = $row['scope'];
        $client->homePage = $row['home_page'];
        $client->showInHome = $row['show_in_home'];

        // Load authentication methods info
        $queryAuthNMethods = "SELECT auth_method FROM oauth2_client_auth_methods WHERE client_id = :id";

        $stmtAuthNMethods = $dbh->prepare($queryAuthNMethods);
        $stmtAuthNMethods->bindParam(':id', $client->id);
        $stmtAuthNMethods->execute();

        foreach ($stmtAuthNMethods->fetchAll() as $row) {
            array_push($client->authMethods, $row['auth_method']);
        }

        // Load scope-audience mapping
        $queryAudienceMapping = "SELECT scope, audience FROM oauth2_client_scope_audience_mapping WHERE client_id = :id";
        $stmtAudienceMapping = $dbh->prepare($queryAudienceMapping);
        $stmtAudienceMapping->bindParam(':id', $client->id);
        $stmtAudienceMapping->execute();

        foreach ($stmtAudienceMapping->fetchAll() as $row) {
            $client->scopeAudienceMap[$row['scope']] = $row['audience'];
        }

        return $client;
    }

}
