<?php

/*
 * This file is part of rap
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

/**
 * Parse X.509 certificate extracting serial number, email, name, surname and institution.
 * Because the certificate puts name and surname together, when name and surname are
 * composed by more than two words this class returns a partial result containing
 * possible alternatives.
 */
class X509Data {

    public $email;
    public $institution;
    public $serialNumber;
    // name(s) and surname, space separated
    public $fullName;
    public $name;
    public $surname;
    // list of possible names: this is populated when fullName has more than two words
    public $candidateNames;

    /**
     * Retrieve full name of the person (name+surname) from CN,
     * removing the e-mail if necessary.
     */
    private function parseCN($cn) {
        $cnSplit = explode(" ", $cn);

        $count = count($cnSplit);
        $lastSegment = $cnSplit[$count - 1];

        if (strpos($lastSegment, "@") === false) {
            if ($count < 2) {
                // We need name + surname
                throw new \Exception("Unparsable CN");
            }
            $this->fullName = $cn;
        } else {
            // Last segment is an email
            if ($count < 3) {
                // We need name + surname + email
                throw new \Exception("Unparsable CN");
            }

            // Rebuilding full name removing the email part
            $cnLength = strlen($cn);
            $emailLength = strlen($lastSegment);
            // -1 is for removing also the space between surname and email
            $this->fullName = substr($cn, 0, $cnLength - $emailLength - 1);
        }
    }

    /**
     * Extract data in a simpler way using openssl_x509_parse PHP function.
     * @param type $sslClientCert $_SERVER['SSL_CLIENT_CERT']
     */
    private function parseUsingOpenSSL($sslClientCert) {

        $parsedX509 = openssl_x509_parse($sslClientCert);

        // try extracting email
        if (isset($parsedX509["extensions"]["subjectAltName"])) {
            $AyAlt = explode(":", $parsedX509["extensions"]["subjectAltName"]);
            if ($AyAlt[0] === "email") {
                $this->email = $AyAlt[1];
            }
        }
        if ($this->email === null && isset($parsedX509["subject"]) && isset($parsedX509["subject"]["emailAddress"])) {
            $this->email = $parsedX509["subject"]["emailAddress"];
        }

        $this->serialNumber = $parsedX509["serialNumber"];

        $cn = $parsedX509["subject"]["CN"];
        $this->parseCN($cn);

        $this->institution = $parsedX509["subject"]["O"];
    }

    /**
     * The serial number is too big to be converted from hex to dec using the
     * builtin hexdec function, so BC Math functions are used.
     * Credits: https://stackoverflow.com/a/1273535/771431
     */
    private static function bchexdec($hex) {
        $dec = 0;
        $len = strlen($hex);
        for ($i = 1; $i <= $len; $i++) {
            $dec = bcadd($dec, bcmul(strval(hexdec($hex[$i - 1])), bcpow('16', strval($len - $i))));
        }
        return $dec;
    }

    /**
     * Populate name and surname or candidateNames variables
     */
    private function fillNameAndSurnameOrCandidates() {
        $nameSplit = explode(' ', $this->fullName);

        if (count($nameSplit) === 2) {
            $this->name = $nameSplit[0];
            $this->surname = $nameSplit[1];
        } else {
            $this->candidateNames = [];
            for ($i = 1; $i < count($nameSplit); $i++) {
                $candidateName = "";
                for ($j = 0; $j < $i; $j++) {
                    if ($j > 0) {
                        $candidateName .= ' ';
                    }
                    $candidateName .= $nameSplit[$j];
                }
                $this->candidateNames[] = $candidateName;
            }
        }
    }

    /**
     * This function is called when the user select the correct candidate name.
     * Surname is calculated as the remaining string.
     * @param type $candidateNameIndex the index of the selected element into 
     * the candidateNames list
     */
    public function selectCandidateName($candidateNameIndex) {
        $candidateName = $this->candidateNames[$candidateNameIndex];
        $this->name = $candidateName;
        $this->surname = substr($this->fullName, strlen($candidateName) + 1);
    }

    /**
     * Extract client certificate data needed by RAP from PHP $_SERVER variable.
     * @param type $server PHP $_SERVER variable. This is passed as parameter in order
     * to make class testable (mocking the $_SERVER variable). Tests have not
     * been written yet.
     * @return \RAP\X509Data
     */
    public static function parse($server) {

        $parsedData = new X509Data();

        if (isset($server['SSL_CLIENT_CERT'])) {
            $parsedData->parseUsingOpenSSL($server['SSL_CLIENT_CERT']);
        }

        if ($parsedData->fullName === null) {
            if (isset($server['SSL_CLIENT_S_DN_CN'])) {
                $parsedData->parseCN($server['SSL_CLIENT_S_DN_CN']);
            } else {
                throw new \Exception("Unable to retrieve CN from certificate");
            }
        }

        if ($parsedData->email === null) {
            if (isset($server['SSL_CLIENT_SAN_Email_0'])) {
                $parsedData->email = $server['SSL_CLIENT_SAN_Email_0'];
            } else {
                throw new \Exception("Unable to retrieve e-mail address from certificate");
            }
        }

        if ($parsedData->serialNumber === null) {
            if (isset($server['SSL_CLIENT_M_SERIAL'])) {
                // In this server variable the serial number is stored into an HEX format,
                // while openssl_x509_parse function provides it in DEC format.
                // Here a hex->dec conversion is performed, in order to store the
                // serial number in a consistent format:
                $hexSerial = $server['SSL_CLIENT_M_SERIAL'];
                $parsedData->serialNumber = X509Data::bchexdec($hexSerial);
            } else {
                throw new Exception("Unable to retrieve serial number from certificate");
            }
        }

        if ($parsedData->institution === null && isset($server['SSL_CLIENT_S_DN_O'])) {
            $parsedData->institution = $server['SSL_CLIENT_S_DN_O'];
        }

        $parsedData->fillNameAndSurnameOrCandidates();

        return $parsedData;
    }

    public function toIdentity() {

        $identity = new Identity(Identity::X509);
        $identity->email = $this->email;
        $identity->name = $this->name;
        $identity->surname = $this->surname;
        $identity->typedId = $this->serialNumber;
        $identity->institution = $this->institution;

        return $identity;
    }

}
