<?php

/*
 * This file is part of rap
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

use PHPUnit\Framework\TestCase;

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

final class LoginHandlerTest extends TestCase {

    private $locatorStub;
    private $userDaoStub;
    private $sessionStub;
    private $userHandlerStub;
    private $auditLogger;
    private $loginHandler;

    public function setUp(): void {
        $this->locatorStub = $this->createMock(\RAP\Locator::class);
        $this->locatorStub->method('getBasePath')->willReturn('http://rap-ia2');

        $this->userDaoStub = $this->createMock(\RAP\UserDAO::class);
        $this->locatorStub->method('getUserDAO')->willReturn($this->userDaoStub);

        $this->sessionStub = $this->createMock(\RAP\SessionData::class);
        $this->locatorStub->method('getSession')->willReturn($this->sessionStub);

        $this->userHandlerStub = $this->createMock(\RAP\UserHandler::class);
        $this->locatorStub->method('getUserHandler')->willReturn($this->userHandlerStub);

        $this->auditLogger = $this->createMock(\Monolog\Logger::class);
        $this->locatorStub->method('getAuditLogger')->willReturn($this->auditLogger);

        $this->loginHandler = new \RAP\LoginHandler($this->locatorStub);
    }

    public function testExistingUserLogin(): void {

        $user = $this->getFakeUser();

        $this->userDaoStub->method('findUserByIdentity')->willReturn($user);

        $this->sessionStub->method('getAction')->willReturn('account');

        $this->sessionStub->method('getUser')->willReturn($user);

        $redirect = $this->loginHandler->onIdentityDataReceived($this->getFakeIdentity1());

        $this->assertEquals('http://rap-ia2/account', $redirect);
    }

    public function testShowConfirmJoinNewIdentity(): void {

        $user = $this->getFakeUser();

        $this->sessionStub->method('getUser')->willReturn($user);
        $this->sessionStub->method('getAction')->willReturn('join');

        $redirect = $this->loginHandler->onIdentityDataReceived($this->getFakeIdentity2());

        $this->assertEquals('http://rap-ia2/confirm-join', $redirect);
    }

    public function testShowConfirmJoinExistingIdentity(): void {

        $user1 = $this->getFakeUser();

        $user2 = $this->getFakeUser();
        $user2->id = '2';
        $user2->identities[0]->id = '5';
        $user2->identities[0]->typedId = '456';

        $this->sessionStub->method('getUser')->willReturn($user1);
        $this->sessionStub->method('getAction')->willReturn('join');

        $this->userDaoStub->method('findUserByIdentity')->willReturn($user2);

        $redirect = $this->loginHandler->onIdentityDataReceived($this->getFakeIdentity2());

        $this->assertEquals('http://rap-ia2/confirm-join', $redirect);
    }

    public function testRegisterFailWithoutSession(): void {
        $this->expectException(\RAP\BadRequestException::class);
        $this->loginHandler->register();
    }

    public function testConfirmJoinFailWithoutSession(): void {
        $this->expectException(\RAP\BadRequestException::class);
        $this->loginHandler->confirmJoin();
    }

    public function testRejectJoinFailWithoutSession(): void {
        $this->expectException(\RAP\BadRequestException::class);
        $this->loginHandler->rejectJoin();
    }

    private function getFakeUser(): \RAP\User {

        $user = new \RAP\User();
        $user->id = '1';
        $identity = new \RAP\Identity('eduGAIN');
        $identity->email = 'test@example.com';
        $identity->id = '4';
        $identity->typedId = '123';
        $user->addIdentity($identity);
        return $user;
    }

    private function getFakeIdentity1(): \RAP\Identity {

        $identity = new \RAP\Identity(\RAP\Identity::EDU_GAIN);
        $identity->typedId = '123';
        $identity->email = 'test@example.com';
        return $identity;
    }

    private function getFakeIdentity2(): \RAP\Identity {

        $identity = new \RAP\Identity(\RAP\Identity::EDU_GAIN);
        $identity->typedId = '456';
        $identity->email = 'test3@example.com';
        return $identity;
    }

}
