<?php

namespace RAP;

use \Firebase\JWT\JWT;

class IdTokenBuilder {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function getIdToken(AccessToken $accessToken, string $nonce = null): string {

        $keyPair = $this->locator->getJWKSDAO()->getNewestKeyPair();

        $payload = $this->createPayloadArray($accessToken, $nonce);

        return JWT::encode($payload, $keyPair->privateKey, $keyPair->alg, $keyPair->keyId);
    }

    private function createPayloadArray(AccessToken $accessToken, string $nonce = null) {
        
        $user = $this->locator->getUserDAO()->findUserById($accessToken->userId);

        $payloadArr = array(
            'iss' => $this->locator->config->jwtIssuer,
            'sub' => strval($user->id),
            'iat' => intval($accessToken->creationTime),
            'exp' => intval($accessToken->expirationTime),
            'name' => $user->getCompleteName(),
            'aud' => $accessToken->clientId
        );

        if ($nonce !== null) {
            $payloadArr['nonce'] = $nonce;
        }

        if (in_array("email", $accessToken->scope)) {
            $payloadArr['email'] = $user->getPrimaryEmail();
        }
        if (in_array("profile", $accessToken->scope)) {
            $payloadArr['given_name'] = $user->getName();
            $payloadArr['family_name'] = $user->getSurname();
            if ($user->getInstitution() !== null) {
                $payloadArr['org'] = $user->getInstitution();
            }
        }

        if ($accessToken->joinUser !== null) {
            $payloadArr['alt_sub'] = strval($accessToken->joinUser);
        }

        return $payloadArr;
    }

}
