<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Perform operations on users.
 */
class UserHandler {

    private $locator;
    private $userDAO;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
        $this->userDAO = $locator->getUserDAO();
    }

    /**
     * Update user information into the database, creating a new user or adding
     * new identities to it.
     * @param \RAP\User $user
     */
    public function saveUser(User $user) {

        $primarySpecified = true;

        // If new user
        if ($user->id === null) {
            $primarySpecified = false;
            $user->id = $this->userDAO->createUser();
        }

        foreach ($user->identities as $identity) {
            if ($identity->id === null) {
                $identity->id = $this->userDAO->insertIdentity($identity, $user->id);
                if (!$primarySpecified) {
                    $this->userDAO->setPrimaryIdentity($user->id, $identity->id);
                    $identity->primary = true;
                }
            }
        }
    }

    /**
     * Build an URL for the web service endpoint that needs to be called in order
     * to move groups from one user to the other during a join operation.
     * @return string grouper URL for the PrepareToJoinServlet
     */
    private function getJoinURL() {
        $joinURL = $this->grouperConfig['wsURL'];

        if (substr($joinURL, -1) !== '/') {
            $joinURL .= '/';
        }
        $joinURL .= 'ia2join';

        return $joinURL;
    }

    public function joinUsers(User $user1, User $user2): User {

        $userId1 = $user1->id;
        $userId2 = $user2->id;

        // Call Grouper for moving groups and privileges from one user to the other
        if (isset($this->locator->config->gmsConfig)) {

            // TODO: change with new GMS
            //create cURL connection
            $conn = curl_init($this->getJoinURL());

            //set options
            curl_setopt($conn, CURLOPT_CONNECTTIMEOUT, 30);
            curl_setopt($conn, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($conn, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($conn, CURLOPT_FOLLOWLOCATION, 1);
            curl_setopt($conn, CURLOPT_USERPWD, $this->grouperConfig['user'] . ":" . $this->grouperConfig['password']);

            //set data to be posted
            curl_setopt($conn, CURLOPT_POST, 1);
            curl_setopt($conn, CURLOPT_POSTFIELDS, "subject1Id=RAP:$userId1&subject2Id=RAP:$userId2");

            //perform the request
            $response = curl_exec($conn);
            $info = curl_getinfo($conn);

            if ($info['http_code'] === 200) {
                curl_close($conn);
            } else {
                //show information regarding the error
                curl_close($conn);
                http_response_code(500);
                die('Error: Grouper response code: ' . $info['http_code']);
            }
        }

        // Call DAO for performing join operation into the RAP database.
        $this->userDAO->joinUsers($userId1, $userId2);

        foreach ($user2->identities as $identity) {
            $identity->primary = false;
            $user1->addIdentity($identity);
        }

        // merged user
        return $user1;
    }

}
