<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Data Access Object interface for accessing the RAP database.
 * Current implementations: RAP\MySQLDAO
 */
interface DAO {

    /**
     * @return type PDO object for accessing the database
     */
    function getDBHandler();

    /**
     * Store a new login token into the database.
     * @param type $token login token
     * @param type $userId
     */
    function createLoginToken($token, $userId);

    /**
     * Retrieve the user ID from the login token.
     * @param type $token
     * @return type user ID
     */
    function findLoginToken($token);

    /**
     * Delete a login token from the database. This happens when the caller
     * application has received the token and used it for retrieving user
     * information from the token using the RAP REST web service.
     * @param type $token login token
     */
    function deleteLoginToken($token);

    /**
     * Create a new identity.
     * @param type $userId the user ID associated to that identity
     * @return type the new identity ID
     */
    function insertIdentity(Identity $identity, $userId);

    /**
     * Create a new user.
     * @return the new user ID.
     */
    function createUser();

    /**
     * @return RAP\User an user object, null if nothing was found.
     */
    function findUserById($userId);

    function setPrimaryIdentity($userId, $identityId);

    /**
     * Retrieve the user associated to a given identity using the typedId.
     * @param type $type Identity type (EDU_GAIN, X509, GOOGLE, ...)
     * @param type $typedId typed unique value used to search the identity in the database
     * @return RAP\User an user object, null if nothing was found.
     */
    function findUserByIdentity($type, $typedId);

    /**
     * Retrieve a set of users matching a given search text.
     * @param type $searchText name, surname or email
     * @return list of RAP\User objects
     */
    function searchUser($searchText);

    /**
     * Store into the database information about a new join request.
     * @param type $token join token
     * @param type $applicantUserId the user asking for the join
     * @param type $targetUserId the user target of the join
     */
    function createJoinRequest($token, $applicantUserId, $targetUserId);

    /**
     * Retrieve join request information.
     * @param type $token join token
     * @return an array of 2 elements having the applicant user id at the first
     * position and the target user id at the second position; null if nothing
     * was found.
     * @throws Exception if multiple requests has been found for the same token.
     */
    function findJoinRequest($token);

    /**
     * Perform a join request.
     * @param type $userId1 the user that will receive all identities
     * @param type $userId2 the user that will lost the identities and will be
     * deleted from the database
     */
    function joinUsers($userId1, $userId2);

    /**
     * When a join action is performed the join request data (join token and user
     * identifiers) needs to be removed from the database.
     * @param type $token join token
     */
    function deleteJoinRequest($token);

    /**
     * CRUD methods for OAuth2Clients (used by admin interface).
     */
    function getOAuth2Clients();

    function createOAuth2Client($client);

    function updateOAuth2Client($client);

    function deleteOAuth2Client($clientId);

    /**
     * Retrieve the client from the configured client id (the one associated to
     * the secret, not the database id).
     */
    function getOAuth2ClientByClientId($clientId);
}
