<?php

namespace RAP;

/**
 * Class implementing the locator pattern in order to implement a rough dependency injection.
 */
class Locator {

    public $config;
    private $serviceLogger;
    private $auditLogger;
    private $dao;
    private $session;
    private $version;

    public function __construct($config) {
        $this->config = $config;

        $this->setupLoggers();
        $this->setupDAO();
        $this->version = file_get_contents(ROOT . '/version.txt');
    }

    public function getVersion(): string {
        return $this->version;
    }

    public function getProtocol(): string {
        return stripos($_SERVER['SERVER_PROTOCOL'], 'https') ? 'https://' : 'http://';
    }

    public function getBasePath(): string {
        return $this->getProtocol() . $_SERVER['HTTP_HOST'] . $this->config->contextRoot;
    }

    public function getDAO(): DAO {
        return $this->dao;
    }

    public function getCallbackHandler(): CallbackHandler {
        return new \RAP\CallbackHandler($dao, $this->getBasePath());
    }

    public function getUserHandler(): UserHandler {
        return new \RAP\UserHandler($this->dao);
    }

    public function getMailSender(): MailSender {
        return new \RAP\MailSender($_SERVER['HTTP_HOST'], $this->getBasePath());
    }

    /**
     * Retrieve the SessionData object from the $_SESSION PHP variable. Create a
     * new one if it is necessary.
     */
    public function getSession(): SessionData {
        if (isset($_SESSION[\RAP\SessionData::KEY])) {
            $this->session = $_SESSION[SessionData::KEY];
        } else {
            $this->session = new \RAP\SessionData();
            $this->session->save();
        }
        return $this->session;
    }

    public function getServiceLogger() {
        return $this->serviceLogger;
    }

    public function getAuditLogger() {
        return $this->auditLogger;
    }

    private function setupLoggers() {
        // Monolog require timezone to be set
        date_default_timezone_set($this->config->timeZone);

        $logLevel = array_search($this->config->logLevel, \Monolog\Logger::getLevels());

        $this->serviceLogger = new \Monolog\Logger('serviceLogger');
        $this->serviceLogger->pushHandler(new \Monolog\Handler\StreamHandler($this->config->serviceLogFile, $logLevel));

        $this->auditLogger = new \Monolog\Logger('auditLogger');
        $this->auditLogger->pushHandler(new \Monolog\Handler\StreamHandler($this->config->auditLogFile, $logLevel));
    }

    private function setupDAO() {
        $databaseConfig = $this->config->databaseConfig;
        switch ($databaseConfig->dbtype) {
            case 'MySQL':
                $this->dao = new \RAP\MySQLDAO($databaseConfig);
                break;
            default:
                throw new Exception($databaseConfig->dbtype . ' not supported yet');
        }
    }

}
