<?php

namespace RAP;

use \Firebase\JWT\JWT;

class TokenBuilder {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function getIdToken(AccessTokenData $tokenData, \Closure $jwtCustomizer = null): string {

        $keyPair = $this->locator->getJWKSDAO()->getNewestKeyPair();

        $payload = $this->createIdTokenPayloadArray($tokenData, $jwtCustomizer);

        return JWT::encode($payload, $keyPair->privateKey, $keyPair->alg, $keyPair->keyId);
    }

    private function createIdTokenPayloadArray(AccessTokenData $tokenData, \Closure $jwtCustomizer = null) {

        $user = $this->locator->getUserDAO()->findUserById($tokenData->userId);

        $payloadArr = array(
            'iss' => $this->locator->config->jwtIssuer,
            'sub' => strval($user->id),
            'iat' => intval($tokenData->creationTime),
            'exp' => intval($tokenData->expirationTime),
            'name' => $user->getCompleteName(),
            'aud' => $tokenData->clientId
        );

        if (in_array("email", $tokenData->scope)) {
            $payloadArr['email'] = $user->getPrimaryEmail();
        }
        if (in_array("profile", $tokenData->scope)) {
            $payloadArr['given_name'] = $user->getName();
            $payloadArr['family_name'] = $user->getSurname();
            if ($user->getInstitution() !== null) {
                $payloadArr['org'] = $user->getInstitution();
            }
        }

        if ($jwtCustomizer !== null) {
            // Add additional custom claims
            $jwtCustomizer($payloadArr);
        }

        return $payloadArr;
    }

    public function getAccessToken(AccessTokenData $tokenData, \Closure $jwtCustomizer = null) {

        $keyPair = $this->locator->getJWKSDAO()->getNewestKeyPair();

        $user = $this->locator->getUserDAO()->findUserById($tokenData->userId);

        $payload = array(
            'iss' => $this->locator->config->jwtIssuer,
            'sub' => strval($user->id),
            'iat' => intval($tokenData->creationTime),
            'exp' => intval($tokenData->expirationTime),
            'aud' => $this->getAudience($tokenData),
            'scope' => implode(' ', $tokenData->scope)
        );
        if ($jwtCustomizer !== null) {
            // Add additional custom claims
            $jwtCustomizer($payload);
        }

        return JWT::encode($payload, $keyPair->privateKey, $keyPair->alg, $keyPair->keyId);
    }

    private function getAudience(AccessTokenData $tokenData) {

        $client = $this->locator->getOAuth2ClientDAO()->getOAuth2ClientByClientId($tokenData->clientId);

        $audiences = [$tokenData->clientId];

        foreach ($tokenData->scope as $scope) {
            if (array_key_exists($scope, $client->scopeAudienceMap)) {
                $audience = $client->scopeAudienceMap[$scope];
                if (!in_array($audience, $audiences)) {
                    array_push($audiences, $audience);
                }
            }
        }

        if (count($audiences) === 1) {
            // according to RFC 7519 audience can be a single value or an array
            return $audiences[0];
        }
        return $audiences;
    }

    /**
     * @param int $lifespan in hours
     * @param string $audience target service
     */
    public function generateNewToken(int $lifespan, string $audience) {
        $keyPair = $this->locator->getJWKSDAO()->getNewestKeyPair();

        $user = $this->locator->getSession()->getUser();

        $iat = time();
        $exp = $iat + $lifespan * 3600;

        $payload = array(
            'iss' => $this->locator->config->jwtIssuer,
            'sub' => strval($user->id),
            'iat' => $iat,
            'exp' => $exp,
            'aud' => $audience
        );

        return JWT::encode($payload, $keyPair->privateKey, $keyPair->alg, $keyPair->keyId);
    }

}
