<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Data model for the user. An user is a set of identities.
 */
class User {

    // User ID
    public $id;
    // List of identities
    public $identities;

    public function __construct() {
        $this->identities = [];
    }

    public function addIdentity(Identity $identity): void {
        array_push($this->identities, $identity);
    }

    public function getPrimaryEmail() {
        foreach ($this->identities as $identity) {
            if ($identity->primary) {
                return $identity->email;
            }
        }
        // A primary identity MUST be defined
        throw new \Exception("No primary identity defined for user " . $this->id);
    }

    /**
     * Returns name and surname if they are present, preferring the primary identity data.
     */
    public function getCompleteName(): ?string {

        $completeName = null;

        foreach ($this->identities as $identity) {
            if ($identity->name !== null && $identity->surname !== null) {
                $completeName = $identity->name . ' ' . $identity->surname;
            }
            if ($identity->primary && $completeName !== null) {
                break;
            }
        }

        return $completeName;
    }

    /**
     * Returns the user name if it is present, preferring the primary identity data.
     */
    public function getName(): ?string {

        $name = null;

        foreach ($this->identities as $identity) {
            if ($identity->name !== null) {
                $name = $identity->name;
            }
            if ($identity->primary && $name !== null) {
                break;
            }
        }

        return $name;
    }

    /**
     * Returns the user surname if it is present, preferring the primary identity data.
     */
    public function getSurname(): ?string {

        $surname = null;

        foreach ($this->identities as $identity) {
            if ($identity->surname !== null) {
                $surname = $identity->surname;
            }
            if ($identity->primary && $surname !== null) {
                break;
            }
        }

        return $surname;
    }

    /**
     * Returns the user institution if it is present, preferring the primary identity data.
     */
    public function getInstitution(): ?string {

        $institution = null;

        foreach ($this->identities as $identity) {
            if ($identity->institution !== null) {
                $institution = $identity->institution;
            }
            if ($identity->primary && $institution !== null) {
                break;
            }
        }

        return $institution;
    }

}
