<?php

/**
 * Front Controller using http://flightphp.com/
 * In all these calls user session must exist, so we have to start it at the 
 * beginning using the startSession() function.
 */
//

function setCallback($callback) {
    global $session, $callbackHandler;

    if (!isset($callback) || $callback === '') {
        $callback = null;
    }
    $session->setCallbackURL($callbackHandler, $callback);
    return $session->getCallbackURL();
}

/**
 * Display the main page (authentication method selection) or the available
 * services list if a valid callback is not found
 */
Flight::route('/', function() {
    startSession();
    $callback = setCallback(Flight::request()->data['callback']);
    global $session, $callbackHandler, $BASE_PATH, $AUTHENTICATION_METHODS, $VERSION;
    if ($callback === null && $session->user === null) {
        Flight::render('services-list.php', array('title' => 'RAP',
            'version' => $VERSION,
            'action' => $BASE_PATH . '/'));
    } else if ($callback !== null && $session->user !== null) {
        $redirectURL = $callbackHandler->getLoginWithTokenURL($session->user->id, $callback);
        Flight::redirect($redirectURL);
    } else {
        Flight::render('index.php', array('title' => 'RAP',
            'version' => $VERSION,
            'session' => $session, 'auth' => $AUTHENTICATION_METHODS));
    }
});

Flight::route('GET /logout', function() {
    startSession();
    session_destroy();
    Flight::redirect('/');
});

function sendAuthRedirect($url) {
    startSession();
    // reload callback from query to avoid problem with session shared between 
    // multiple browser tabs
    setCallback(Flight::request()->query['callback']);
    Flight::redirect($url);
}

Flight::route('/google', function() {
    sendAuthRedirect('/auth/oauth2/google_token.php');
});

Flight::route('/facebook', function() {
    sendAuthRedirect('/auth/oauth2/facebook_login.php');
});

Flight::route('/linkedIn', function() {
    sendAuthRedirect('/auth/oauth2/linkedin_login.php');
});

Flight::route('/eduGAIN', function() {
    sendAuthRedirect('/auth/saml2/aai.php');
});

Flight::route('/x509', function() {
    sendAuthRedirect('/auth/x509/certlogin.php');
});

Flight::route('/direct', function() {
    global $AUTHENTICATION_METHODS;
    sendAuthRedirect($AUTHENTICATION_METHODS['DirectIdP']['url']);
});

/**
 * Render the join confirmation page (confirmation link received by email).
 */
Flight::route('GET /confirm-join', function() {
    $token = Flight::request()->query['token'];

    if ($token === null) {
        http_response_code(422);
        die("Token not found");
    }

    global $dao, $VERSION;

    $userIds = $dao->findJoinRequest($token);
    if ($userIds === null) {
        http_response_code(422);
        die("Invalid token");
    }

    $applicantUser = $dao->findUserById($userIds[0]);
    $targetUser = $dao->findUserById($userIds[1]);

    Flight::render('confirm-join.php', array('title' => 'RAP',
        'version' => $VERSION,
        'token' => $token,
        'applicantUser' => $applicantUser,
        'targetUser' => $targetUser));
});

/**
 * Confirm a join and show the page containing the operation status.
 */
Flight::route('POST /confirm-join', function() {

    global $dao, $userHandler, $auditLog;

    $token = Flight::request()->data['token'];

    if ($token === null) {
        http_response_code(422);
        die("Token not found");
    }

    $userIds = $dao->findJoinRequest($token);
    if ($userIds === null) {
        http_response_code(422);
        die("Invalid token");
    }

    $auditLog->info("JOIN," . $userIds[0] . "," . $userIds[1]);

    $userHandler->joinUsers($userIds[0], $userIds[1]);
    $dao->deleteJoinRequest($token);

    // Force user to relogin to see changes to him/her identities
    session_start();
    session_destroy();

    global $BASE_PATH, $VERSION;
    Flight::render('join-success.php', array('title' => 'Success - RAP Join Request',
        'version' => $VERSION,
        'basePath' => $BASE_PATH));
});

/**
 * Render the page for selecting th correct name and username from candidates
 * list during a X.509 registration.
 */
Flight::route('GET /x509-name-surname', function() {

    startSession();
    global $session, $BASE_PATH, $VERSION;

    if ($session->x509DataToRegister !== null && $session->x509DataToRegister->name === null) {
        Flight::render('x509-name-surname.php', array('title' => 'Select name and surname',
            'version' => $VERSION,
            'fullName' => $session->x509DataToRegister->fullName,
            'candidateNames' => $session->x509DataToRegister->candidateNames));
    } else {
        // Redirect to index
        header("Location: " . $BASE_PATH);
        die();
    }
});

/**
 * Complete the X.509 registration selecting the correct name and surname specified
 * by the user.
 */
Flight::route('POST /submit-x509-name', function() {

    $selectedNameIndex = Flight::request()->data['selected-name'];
    
    error_log('index=' . $selectedNameIndex);
    
    startSession();
    global $session, $BASE_PATH;
    
    if ($session->x509DataToRegister !== null) {
        $session->x509DataToRegister->selectCandidateName($selectedNameIndex);
        $session->userToLogin = $session->x509DataToRegister->toUser();
        $session->x509DataToRegister = null;
        $session->save();
        header("Location: " . $BASE_PATH . '/tou-check');
        die();
    } else {
        die('X.509 data not returned');
    }
});

Flight::route('GET /tou-check', function() {

    startSession();
    global $session, $BASE_PATH, $VERSION;

    if ($session->userToLogin === null) {
        die("User data not retrieved.");
    } else {
        Flight::render('tou-check.php', array('title' => 'Terms of Use acceptance',
            'user' => $session->userToLogin,
            'version' => $VERSION,
            'registration_url' => $BASE_PATH . '/register'));
    }
});

Flight::route('GET /register', function() {

    startSession();
    global $session, $userHandler, $auditLog, $callbackHandler;

    if ($session->userToLogin === null) {
        die("User data not retrieved.");
    } else {

        $user = $session->userToLogin;
        $userHandler->saveUser($user);

        $session->userToLogin = null;
        $session->save();

        $auditLog->info("LOGIN," . $user->identities[0]->type . "," . $user->id);
        $callbackHandler->manageLoginRedirect($user, $session);
    }
});
