<?php

/*
 * This file is part of rap
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

class MySQLRefreshTokenDAO extends BaseMySQLDAO implements RefreshTokenDAO {

    public function __construct(Locator $locator) {
        parent::__construct($locator);
    }

    function createRefreshTokenData(RefreshTokenData $refreshTokenData): RefreshTokenData {

        $dbh = $this->getDBHandler();
        $stmt = $dbh->prepare("INSERT INTO refresh_token (token_hash, user_id, client_id, scope, creation_time, expiration_time)"
                . " VALUES(:token_hash, :user_id, :client_id, :scope, :creation_time, :expiration_time)");

        $scope = null;
        if ($refreshTokenData->scope !== null) {
            $scope = join(' ', $refreshTokenData->scope);
        }

        $params = array(
            ':token_hash' => $refreshTokenData->tokenHash,
            ':user_id' => $refreshTokenData->userId,
            ':client_id' => $refreshTokenData->clientId,
            ':scope' => $scope,
            ':creation_time' => $refreshTokenData->creationTime,
            ':expiration_time' => $refreshTokenData->expirationTime
        );

        if ($stmt->execute($params)) {
            return $refreshTokenData;
        } else {
            error_log($stmt->errorInfo()[2]);
            throw new \Exception("SQL error while storing user token");
        }
    }

    function getRefreshTokenData(string $tokenHash): ?RefreshTokenData {

        $dbh = $this->getDBHandler();

        $stmt = $dbh->prepare("SELECT user_id, client_id, creation_time, expiration_time, scope "
                . " FROM refresh_token WHERE token_hash = :token_hash");

        $stmt->bindParam(':token_hash', $tokenHash);

        $stmt->execute();

        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }

        $token = new RefreshTokenData();
        $token->tokenHash = $tokenHash;
        $token->userId = $row['user_id'];
        $token->clientId = $row['client_id'];
        $token->creationTime = $row['creation_time'];
        $token->expirationTime = $row['expiration_time'];

        $scope = null;
        if (isset($row['scope'])) {
            $scope = $row['scope'];
        }
        if ($scope !== null && $scope !== '') {
            $token->scope = explode(' ', $scope);
        }

        return $token;
    }

    function deleteRefreshTokenData(string $tokenHash): void {

        $dbh = $this->getDBHandler();

        $stmt = $dbh->prepare("DELETE FROM refresh_token WHERE token_hash = :token_hash");
        $stmt->bindParam(':token_hash', $tokenHash);
        $stmt->execute();
    }

}
