<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

class GrouperClient {

    private $client;

    function __construct($config) {

        $this->client = new \SoapClient($config['wsdlURL'], array(
            'login' => $config['user'],
            'password' => $config['password'],
            'trace' => 1,
            // See: https://bugs.php.net/bug.php?id=36226
            'features' => SOAP_SINGLE_ELEMENT_ARRAYS
                )
        );
    }

    private function getBaseRequestParams() {
        return array(
            'clientVersion' => 'v2_3_000'
        );
    }

    private function startsWith($haystack, $needle) {
        return strpos($haystack, "$needle", 0) === 0;
    }

    private function isSuccess($response) {
        $success = isset($response->return->resultMetadata) && $response->return->resultMetadata->resultCode === 'SUCCESS';
        if (!$success) {
            throw new \Exception("Web Service Failure. Response=" . json_encode($response));
        }
        return $success;
    }

    public function getSubjectGroups($subjectId) {

        $params = $this->getBaseRequestParams();
        $params['subjectLookups'] = array(
            'subjectId' => $subjectId,
            'subjectSourceId' => 'RAP'
        );

        $response = $this->client->getGroups($params);

        if ($this->isSuccess($response)) {
            if (count($response->return->results) === 1) {
                $groups = [];
                if ($response->return->results[0]->wsGroups !== null) {
                    foreach ($response->return->results[0]->wsGroups as $group) {
                        if (!$this->startsWith($group->name, 'etc:')) {
                            array_push($groups, $group->name);
                        }
                    }
                }
                return $groups;
            } else {
                throw new \Exception("Wrong results number. Response=" . json_encode($response));
            }
        }
    }

    public function addMemberships($subjectId, $groups) {

        foreach ($groups as $group) {
            $params = $this->getBaseRequestParams();
            $params['subjectId'] = $subjectId;
            $params['subjectSourceId'] = 'RAP';
            $params['groupName'] = $group;

            $this->client->addMemberLite($params);
        }
    }

    public function removeMemberships($subjectId, $groups) {

        foreach ($groups as $group) {
            $params = $this->getBaseRequestParams();
            $params['subjectId'] = $subjectId;
            $params['subjectSourceId'] = 'RAP';
            $params['groupName'] = $group;

            $this->client->deleteMemberLite($params);
        }
    }

}
