<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

class X509Data {

    public $email;
    public $fullName;
    public $institution;
    public $serialNumber;
    public $name;
    public $surname;
    public $candidateNames;

    /**
     * Retrieve full name from CN, removing the e-mail if necessary
     */
    private function parseCN($cn) {
        $cnSplit = explode(" ", $cn);

        $count = count($cnSplit);
        $lastSegment = $cnSplit[$count - 1];

        if (strpos($lastSegment, "@") === false) {
            if ($count < 2) {
                // We need name + surname
                throw new \Exception("Unparsable CN");
            }
            $this->fullName = $cn;
        } else {
            // Last segment is an email
            if ($count < 3) {
                // We need name + surname + email
                throw new \Exception("Unparsable CN");
            }

            // Rebuilding full name removing the email part
            $cnLength = strlen($cn);
            $emailLength = strlen($lastSegment);
            // -1 is for removing also the space between surname and email
            $this->fullName = substr($cn, 0, $cnLength - $emailLength - 1);
        }
    }

    private function parseUsingOpenSSL($sslClientCert) {

        $parsedX509 = openssl_x509_parse($sslClientCert);

        // try extracting email
        if (isset($parsedX509["extensions"]["subjectAltName"])) {
            $AyAlt = explode(":", $parsedX509["extensions"]["subjectAltName"]);
            if ($AyAlt[0] === "email") {
                $this->email = $AyAlt[1];
            }
        }

        $this->serialNumber = $parsedX509["serialNumber"];

        $cn = $parsedX509["subject"]["CN"];
        $this->parseCN($cn);

        $this->institution = $parsedX509["subject"]["O"];
    }

    /**
     * Credits: https://stackoverflow.com/a/1273535/771431
     */
    private static function bchexdec($hex) {
        $dec = 0;
        $len = strlen($hex);
        for ($i = 1; $i <= $len; $i++) {
            $dec = bcadd($dec, bcmul(strval(hexdec($hex[$i - 1])), bcpow('16', strval($len - $i))));
        }
        return $dec;
    }

    private function fillNameAndSurnameOrCandidates() {
        $nameSplit = explode(' ', $this->fullName);

        if (count($nameSplit) === 2) {
            $this->name = $nameSplit[0];
            $this->surname = $nameSplit[1];
        } else {
            $this->candidateNames = [];
            for ($i = 1; $i < count($nameSplit); $i++) {
                $candidateName = "";
                for ($j = 0; $j < $i; $j++) {
                    if ($j > 0) {
                        $candidateName .= ' ';
                    }
                    $candidateName .= $nameSplit[$j];
                }
                $this->candidateNames[] = $candidateName;
            }
        }
    }

    public function selectCandidateName($candidateNameIndex) {
        $candidateName = $this->candidateNames[$candidateNameIndex];
        $this->name = $candidateName;
        $this->surname = substr($this->fullName, strlen($candidateName) + 1);
    }

    public static function parse($server) {

        $parsedData = new X509Data();

        if (isset($server['SSL_CLIENT_CERT'])) {
            $parsedData->parseUsingOpenSSL($server['SSL_CLIENT_CERT']);
        }

        if ($parsedData->fullName === null) {
            if (isset($server['SSL_CLIENT_S_DN_CN'])) {
                $parsedData->parseCN($server['SSL_CLIENT_S_DN_CN']);
            } else {
                throw new \Exception("Unable to retrieve CN from certificate");
            }
        }

        if ($parsedData->email === null) {
            if (isset($server['SSL_CLIENT_SAN_Email_0'])) {
                $parsedData->email = $server['SSL_CLIENT_SAN_Email_0'];
            } else {
                throw new \Exception("Unable to retrieve e-mail address from certificate");
            }
        }

        if ($parsedData->serialNumber === null) {
            if (isset($server['SSL_CLIENT_M_SERIAL'])) {
                // In this server variable the serial number is stored into an HEX format,
                // while openssl_x509_parse function provides it in DEC format.
                // Here a hex->dec conversion is performed, in order to store the
                // serial number in a consistent format:
                $hexSerial = $server['SSL_CLIENT_M_SERIAL'];
                $parsedData->serialNumber = X509Data::bchexdec($hexSerial);
            } else {
                throw new Exception("Unable to retrieve serial number from certificate");
            }
        }

        if ($parsedData->institution === null && isset($server['SSL_CLIENT_S_DN_O'])) {
            $parsedData->institution = $server['SSL_CLIENT_S_DN_O'];
        }

        $parsedData->fillNameAndSurnameOrCandidates();

        return $parsedData;
    }

}
