<?php

use PHPUnit\Framework\TestCase;

final class OAuth2RequestHandlerTest extends TestCase {

    public function testBadRequestExceptionIfMissingClientId(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => null
        ];

        $locatorStub = $this->createMock(\RAP\Locator::class);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testInvalidClientRedirectURI(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "redirect_uri",
            "state" => "state",
            "alg" => null,
            "scope" => "email%20profile"
        ];

        $daoStub = $this->createMock(\RAP\OAuth2ClientDAO::class);
        $daoStub->method('getOAuth2ClientByClientId')->willReturn(new \RAP\OAuth2Client());

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getOAuth2ClientDAO')->willReturn($daoStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testExecuteOAuthStateFlow(): void {

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "redirect_uri",
            "state" => "state",
            "alg" => null,
            "scope" => "email%20profile"
        ];

        $daoStub = $this->createMock(\RAP\OAuth2ClientDAO::class);
        $client = new \RAP\OAuth2Client();
        $client->redirectUrl = "redirect_uri";
        $daoStub->method('getOAuth2ClientByClientId')->willReturn($client);

        $sessionStub = $this->createMock(\RAP\SessionData::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getOAuth2ClientDAO')->willReturn($daoStub);
        $locatorStub->method('getSession')->willReturn($sessionStub);

        $sessionStub->expects($this->once())
                ->method('setOAuth2Data')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testExpiresIn(): void {

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $accessToken = new \RAP\AccessToken();

        $expDate = new \DateTime();
        $expDate->add(new \DateInterval('PT1H'));

        $accessToken->expirationTime = $expDate->format("Y-m-d H:i:s");

        // testing private method using reflection
        $reflection = new \ReflectionClass(get_class($requestHandler));
        $method = $reflection->getMethod('getExpiresIn');
        $method->setAccessible(true);

        $exp = $method->invokeArgs($requestHandler, [$accessToken]);

        $this->assertEquals(3600, $exp);
    }

}
