<?php

namespace RAP;

use \Firebase\JWT\JWT;

class OAuth2RequestHandler {

    private $locator;

    public function __construct(\RAP\Locator $locator) {
        $this->locator = $locator;
    }

    public function handleAuthorizeRequest($params) {

        if ($params['client_id'] === null) {
            throw new BadRequestException("Client id is required");
        }

        if ($params['redirect_uri'] === null) {
            throw new BadRequestException("Redirect URI is required");
        }

        $client = $this->locator->getOAuth2ClientDAO()->getOAuth2ClientByClientId($params['client_id']);
        if ($client === null) {
            throw new BadRequestException("Invalid client id: " . $params['client_id']);
        }
        if ($client->redirectUrl !== $params['redirect_uri']) {
            throw new BadRequestException("Invalid client redirect URI: " . $params['redirect_uri']);
        }

        $alg = $params['alg'];
        if ($alg === null) {
            $alg = "RS256";
        }

        $state = $params['state'];
        $nonce = $params['nonce'];

        if ($state === null && $nonce === null) {
            throw new BadRequestException("State or nonce is required");
        }

        // Storing OAuth2 data in session
        $oauth2Data = new \RAP\OAuth2Data();
        $oauth2Data->clientId = $client->client;
        $oauth2Data->redirectUrl = $client->redirectUrl;
        $oauth2Data->state = $state;
        $oauth2Data->nonce = $nonce;

        $scope = $params['scope'];
        if ($scope !== null) {
            $oauth2Data->scope = explode(' ', $scope);
        }

        $session = $this->locator->getSession();
        $session->setOAuth2Data($oauth2Data);
    }

    public function getRedirectResponseUrl(): string {

        $session = $this->locator->getSession();

        $accessToken = new \RAP\AccessToken();
        $accessToken->code = base64_encode(bin2hex(openssl_random_pseudo_bytes(64)));
        $accessToken->token = base64_encode(bin2hex(openssl_random_pseudo_bytes(128)));
        $accessToken->userId = $session->getUser()->id;
        $accessToken->clientId = $session->getOAuth2Data()->clientId;
        $accessToken->redirectUri = $session->getOAuth2Data()->redirectUrl;
        $accessToken->scope = $session->getOAuth2Data()->scope;

        $this->locator->getAccessTokenDAO()->createAccessToken($accessToken);

        $state = $session->getOAuth2Data()->state;
        $nonce = $session->getOAuth2Data()->nonce;

        if ($state !== null) {
            // Authorization code grant flow
            $redirectUrl = $session->getOAuth2Data()->redirectUrl
                    . '?code=' . $accessToken->code . '&scope=profile&state=' . $state;
        } else {
            // Implicit grant flow
            $idToken = $this->locator->getIdTokenBuilder()->getIdToken($accessToken, $nonce);
            $redirectUrl = $session->getOAuth2Data()->redirectUrl . "#id_token=" . $idToken;
        }

        return $redirectUrl;
    }

    public function handleAccessTokenRequest($params): array {

        if ($params['code'] === null) {
            throw new BadRequestException("code id is required");
        }

        if ($params['redirect_uri'] === null) {
            throw new BadRequestException("Redirect URI is required");
        }

        // Note: theorically the standard wants also the client_id here,
        // however some clients don't send it

        $accessToken = $this->locator->getAccessTokenDAO()->retrieveAccessTokenFromCode($params['code']);

        if ($accessToken === null) {
            throw new BadRequestException("No token for given code");
        }

        if ($accessToken->redirectUri !== $params['redirect_uri']) {
            throw new BadRequestException("Invalid redirect URI: " . $params['redirect_uri']);
        }

        return $this->getAccessTokenResponse($accessToken);
    }

    public function handleRefreshTokenRequest($params): array {

        if ($params['refresh_token'] === null) {
            throw new BadRequestException("refresh_token is required");
        }

        $refreshToken = $this->locator->getRefreshTokenDAO()->getRefreshToken($params['refresh_token']);

        if ($refreshToken === null || $refreshToken->isExpired()) {
            throw new UnauthorizedException("Invalid refresh token");
        }

        $scope = $this->getScope($params, $refreshToken);

        // Generating a new access token
        $accessToken = new AccessToken();
        $accessToken->token = base64_encode(bin2hex(openssl_random_pseudo_bytes(128)));
        $accessToken->clientId = $refreshToken->clientId;
        $accessToken->userId = $refreshToken->userId;
        $accessToken->scope = $scope;

        $accessToken = $this->locator->getAccessTokenDAO()->createAccessToken($accessToken);

        return $this->getAccessTokenResponse($accessToken);
    }

    /**
     * We can request a new access token with a scope that is a subset (or the
     * same set) of the scope defined for the refresh token.
     */
    private function getScope(array $params, RefreshToken $refreshToken): ?array {

        $scope = $refreshToken->scope;

        if ($params['scope'] !== null) {

            $newScopeValues = explode(' ', $params['scope']);

            foreach ($newScopeValues as $newScopeValue) {
                $found = false;
                foreach ($scope as $oldScopeValue) {
                    if ($oldScopeValue === $newScopeValue) {
                        $found = true;
                        break;
                    }
                }
                if (!$found) {
                    throw new BadRequestException("Scope " . $newScopeValue . " was not defined for the given refresh token");
                }
            }

            $scope = $newScopeValues;
        }

        return $scope;
    }

    private function getAccessTokenResponse(AccessToken $accessToken) {

        $result = [];
        $result['access_token'] = $accessToken->token;
        $result['token_type'] = 'Bearer';
        $result['expires_in'] = $accessToken->expirationTime - time();
        $result['refresh_token'] = $this->getNewRefreshToken($accessToken);

        if ($accessToken->scope !== null && in_array('openid', $accessToken->scope)) {
            $result['id_token'] = $this->locator->getIdTokenBuilder()->getIdToken($accessToken);
        }

        return $result;
    }

    private function getNewRefreshToken(AccessToken $accessToken): string {

        $refreshToken = new RefreshToken();
        $refreshToken->token = base64_encode(bin2hex(openssl_random_pseudo_bytes(128)));
        $refreshToken->clientId = $accessToken->clientId;
        $refreshToken->userId = $accessToken->userId;
        $refreshToken->scope = $accessToken->scope;

        $this->locator->getRefreshTokenDAO()->createRefreshToken($refreshToken);

        return $refreshToken->token;
    }

    public function handleCheckTokenRequest($token): array {

        $accessToken = $this->locator->getAccessTokenDAO()->getAccessToken($token);
        if ($accessToken === null) {
            throw new UnauthorizedException("Invalid access token");
        }

        $user = $this->locator->getUserDAO()->findUserById($accessToken->userId);

        $result = [];
        $result['exp'] = $accessToken->expirationTime - time();
        $result['user_name'] = $user->id;
        $result['client_id'] = $accessToken->clientId;
        $result['refresh_token'] = $this->getNewRefreshToken($accessToken);

        if ($accessToken->scope !== null) {
            $result['scope'] = $accessToken->scope;
            if (in_array('openid', $accessToken->scope)) {
                $result['id_token'] = $this->locator->getIdTokenBuilder()->getIdToken($accessToken);
            }
        }

        return $result;
    }

    public function validateToken(): void {
        $headers = apache_request_headers();

        if (!isset($headers['Authorization'])) {
            throw new BadRequestException("Missing Authorization header");
        }

        $authorizationHeader = explode(" ", $headers['Authorization']);
        if ($authorizationHeader[0] === "Bearer") {
            $bearer_token = $authorizationHeader[1];
        } else {
            throw new BadRequestException("Invalid token type");
        }

        $accessToken = $this->locator->getAccessTokenDAO()->getAccessToken($bearer_token);
        if ($accessToken === null) {
            $this->attemptJWTTokenValidation($bearer_token);
        } else if ($accessToken->isExpired()) {
            throw new UnauthorizedException("Access token is expired");
        }
    }

    private function attemptJWTTokenValidation($jwt): void {

        $jwtParts = explode('.', $jwt);
        if (count($jwtParts) === 0) {
            throw new UnauthorizedException("Invalid token");
        }

        $header = JWT::jsonDecode(JWT::urlsafeB64Decode($jwtParts[0]));
        if (!isset($header->kid)) {
            throw new UnauthorizedException("Invalid token: missing kid in header");
        }

        $keyPair = $this->locator->getJWKSDAO()->getRSAKeyPairById($header->kid);
        if ($keyPair === null) {
            throw new UnauthorizedException("Invalid kid: no key found");
        }

        try {
            JWT::decode($jwt, $keyPair->publicKey, [$keyPair->alg]);
        } catch (\Firebase\JWT\ExpiredException $ex) {
            throw new UnauthorizedException("Access token is expired");
        }
    }

}
