<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * Perform operations on users.
 */
class UserHandler {

    private $locator;
    private $userDAO;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
        $this->userDAO = $locator->getUserDAO();
    }

    /**
     * Update user information into the database, creating a new user or adding
     * new identities to it.
     * @param \RAP\User $user
     */
    public function saveUser(User $user) {

        $primarySpecified = true;

        // If new user
        if ($user->id === null) {
            $primarySpecified = false;
            $user->id = $this->userDAO->createUser();
        }

        foreach ($user->identities as $identity) {
            if ($identity->id === null) {
                $identity->id = $this->userDAO->insertIdentity($identity, $user->id);
                if (!$primarySpecified) {
                    $this->userDAO->setPrimaryIdentity($user->id, $identity->id);
                    $identity->primary = true;
                }
            }
        }
    }

    public function joinUsers(User $user1, User $user2): User {

        $userId1 = $user1->id;
        $userId2 = $user2->id;

        // Call Grouper for moving groups and privileges from one user to the other
        if (isset($this->locator->config->gms)) {

            // TODO: change with new GMS
            //create cURL connection
            $conn = curl_init($this->locator->config->gms->joinEndpoint);

            //set options
            curl_setopt($conn, CURLOPT_CONNECTTIMEOUT, 30);
            curl_setopt($conn, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($conn, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($conn, CURLOPT_FOLLOWLOCATION, 1);
            curl_setopt($conn, CURLOPT_HTTPHEADER, ['Authorization: Bearer '
                . $this->getJoinIdToken($userId1, $userId2)]);

            //set data to be posted
            curl_setopt($conn, CURLOPT_POST, 1);

            //perform the request
            $response = curl_exec($conn);
            $info = curl_getinfo($conn);

            if ($info['http_code'] === 200) {
                curl_close($conn);
            } else {
                //show information regarding the error
                curl_close($conn);
                error_log($response);
                http_response_code(500);
                die('Error: GMS response code: ' . $info['http_code'] . "\n");
            }
        }

        // Call DAO for performing join operation into the RAP database.
        $this->userDAO->joinUsers($userId1, $userId2);

        foreach ($user2->identities as $identity) {
            $identity->primary = false;
            $user1->addIdentity($identity);
        }

        // merged user
        return $user1;
    }

    private function getJoinIdToken(int $userId1, int $userId2): string {

        $gmsId = $this->locator->config->gms->id;

        $accessToken = new AccessToken();
        $accessToken->clientId = $gmsId;
        $accessToken->userId = $userId1;
        $accessToken->joinUser = $userId2;
        // shorter expiration
        $accessToken->expirationTime = $accessToken->creationTime + 100;
        $accessToken->scope = ['openid'];

        return $this->locator->getIdTokenBuilder()->getIdToken($accessToken);
    }

}
