<?php

/*
 * This file is part of rap
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

class MySQLJWKSDAO extends BaseMySQLDAO implements JWKSDAO {

    public function __construct($config) {
        parent::__construct($config);
    }

    public function insertRSAKeyPair(RSAKeyPair $keyPair): RSAKeyPair {

        $dbh = $this->getDBHandler();

        $query = "INSERT INTO rsa_keypairs(id, private_key, public_key, alg, creation_time) VALUES (:id, :private_key, :public_key, :alg, :creation_time)";

        $now = time();

        $stmt = $dbh->prepare($query);
        $stmt->bindParam(':id', $keyPair->keyId);
        $stmt->bindParam(':private_key', $keyPair->privateKey);
        $stmt->bindParam(':public_key', $keyPair->publicKey);
        $stmt->bindParam(':alg', $keyPair->alg);
        $stmt->bindParam(':creation_time', $now);

        $stmt->execute();

        return $keyPair;
    }

    public function getRSAKeyPairs(): array {

        $dbh = $this->getDBHandler();

        $query = "SELECT id, private_key, public_key, alg, creation_time FROM rsa_keypairs ORDER BY creation_time DESC";

        $stmt = $dbh->prepare($query);
        $stmt->execute();

        $keyPairs = [];
        foreach ($stmt->fetchAll() as $row) {
            $keyPair = $this->getRSAKeyPairFromResultRow($row);
            array_push($keyPairs, $keyPair);
        }

        return $keyPairs;
    }

    public function getRSAKeyPairById(string $id): ?RSAKeyPair {

        $dbh = $this->getDBHandler();

        $query = "SELECT id, private_key, public_key, alg, creation_time FROM rsa_keypairs WHERE id = :id";

        $stmt = $dbh->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();

        foreach ($stmt->fetchAll() as $row) {
            return $this->getRSAKeyPairFromResultRow($row);
        }

        return null;
    }

    public function getNewestKeyPair(): ?RSAKeyPair {
        $dbh = $this->getDBHandler();

        $query = "SELECT id, private_key, public_key, alg, creation_time FROM rsa_keypairs ORDER BY creation_time DESC LIMIT 1";

        $stmt = $dbh->prepare($query);
        $stmt->execute();

        foreach ($stmt->fetchAll() as $row) {
            return $this->getRSAKeyPairFromResultRow($row);
        }

        return null;
    }

    private function getRSAKeyPairFromResultRow(array $row): RSAKeyPair {
        $keyPair = new RSAKeyPair();
        $keyPair->keyId = $row['id'];
        $keyPair->privateKey = $row['private_key'];
        $keyPair->publicKey = $row['public_key'];
        $keyPair->alg = $row['alg'];
        $keyPair->creationTime = $row['creation_time'];
        return $keyPair;
    }

    public function deleteKeyPair(string $id): void {

        $dbh = $this->getDBHandler();

        $query = "DELETE FROM rsa_keypairs WHERE id = :id";

        $stmt = $dbh->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
    }

}
