# README

This file describes the structure of the Metis L1 pipeline. The pipeline source code is written using the Interactive Data Language (IDL). The present version has been developed and tested using IDL 8.7 and higher versions.

## metis_l1_prep.pro
Main pipeline program.

### Call
The main program can be launched from outside IDL with the following command:

	idl -e metis_l1_prep

The main program must be launched from a working directory containing two folders, *input* and *output*, where the data files used and generated by the pipeline are respectively located.

### Framework setup
IDL must include the NASA IDL Astronomy Users's Library, which can be downloaded at the following link: https://idlastro.gsfc.nasa.gov.
In addition, the SPICE toolkit for IDL provided by the NASA's Navigation and Ancillary Information Facility (NAIF; https://naif.jpl.nasa.gov/naif) must be installed.

### Operations
The pipeline performs the following specific operations:
+ **saturation and quality classification** (applied only to image data products): a quality matrix with same size of the input image is created, containing classification of the quality of each pixel
+ **image rebinning** (applied only to image data products): image is rebinned according to the commanded binning, and pixel values are multiplied by the relevant binning factor (4.0 for 2×2 binning and 16.0 for 4×4 binning)
+ **exposure time correction** (applied only to visible-light and UV image data products): pixel values are multiplied by the total number of frames averaged by the Metis on-board software
+ **bad-pixel flagging** (applied only to image data products): the quality matrix is updated to take into account and to provide indication of known corrupted/unreliable detector pixels; the values of the pixels (or bins) in the quality matrix corresponding to (or associated with) a bad pixel are set to zero, according to the convention used for the saturation and quality classification
+ **time conversions**:all times (begin and end of acquisition) are converted from on-board time units to UTC using the last released version of the Solar Orbiter's SPICE kernel dataset
+ **determination of the World Coordinate System (WCS) parameters and spacecraft coordinates**: WCS keywords and spacecraft coordinates are computed using the boresight parameters given in the Metis calibration package and the last released version of the Solar Orbiter's SPICE ephemeris and attitude kernels, and are added to the FITS header

### Input
The pipeline input data are listed in a JSON file located in the *input* folder. The file must be named *contents.json* and formatted according to the following structure:

	{
        "file_name": "",
        "spice_kernels": "",
        "sk_arc_version": "",
        "cal_pack_path": "",
        "planning_file_name": "",
        "hk_file_name": "",
        "sw_version": "",
        "obsw_version": "",
        "idb_version": "",
        "hdr_version": "",
        "l1_version": 0
	}

where all fields are defined as follows:

+ **file_name**: a string scalar containing the full pathname of the L0 FITS file to be processed
+ **spice_kernels**: a string scalar containing the full path of the last released version of the Solar Orbiter's SPICE ephemeris and attitude kernels 
+ **sk_arc_version**: a string scalar containing the three-digit version identifier of the SPICE kernels
+ **cal_pack_path**: a string scalar containing the full path of the applicable version of the Metis calibration package
+ **planning_file_name**: a string scalar containing the full pathname of a JSON file containing the planning metadata associated with the input L0 FITS file
+ **hk_file_name**: a string scalar containing the full pathname of a JSON file containing the set of Metis house-keeping parameters associated with the input L0 data
+ **sw_version**: a string scalar containing the version identifier of the processor used for generating the final pipeline product, i.e. the L1 FITS file
+ **obsw_version**: a string scalar containing the version identifier of the Metis on-board software used for the acquisition of the input data
+ **idb_version**: a string scalar containing the version identifier of the current instrument database used for the processing of the input data
+ **hdr_version**: a string scalar containing the version identifier of the scientific header structure associated with the input data
+ **l1_version**: an integer scalar corresponding to the incremental number version of the last L1 FITS file generated from the same input L0 FITS file

> ***Note***: all input files listed in the *contents.json* file must be located in the *input* folder and their pathname must be relative to the main working directory

### Additional input
#### Planning metadata
The planning metadata JSON file has the following structure:

    {
        "mission_ltp": 0,
        "obs_id": "",
        "soop_name": "",
        "obs_mode": ""
    }

where all fields are defined as follows:
+ **mission_ltp**: an integer scalar corresponding to the LTP period number which the observation belongs to
+ **obs_id**: a string scalar containing the value of the *socIds* key associated with the observation which the data belongs to, contained in the Observation Timeline Export JSON file provided by the Solar Orbiter SOC
> ***Note***: this field must be formatted as `smet_xxxx_xxx_xxx_xxxx_xxx`, where `x` can be either a character or a digit
+ **obs_mode**: a string scalar containing the value of the *name* key that is associated with the observation which the data belongs to, contained in the Observation Timeline Export JSON file 
+ **soop_name**: a string scalar containing the value of the *soopType* key that is associated with the SOOP which the observation belongs to, contained in the Observation Timeline Export JSON file 

#### House-keeping parameters
The JSON file containing the house-keeping parameters associated with the L0 data has the following structure:
    
    {
        "par_name": [""],
        "packet": [""],
        "rec_time": [""],
        "gen_time": [""],
        "raw_val": [""],
        "eng_val": [""],
        "unit": [""],
        "desc": [""]
    }
where all fields are defined as follows:
+ **par_name**: a one-dimensional string array containing the names of the house-keeping parameters
+ **packet**: a one-dimensional string array containing the names of the telemetry packets which the parameters belong to
+ **rec_time**: a one-dimensional string array containing the UTC packet creation times in ISO8601 format
> ***Note***: this is the time at which the SCOS-2000 packet was created, which is also approximately the time at which the packet was received by the MCS and stored
+ **gen_time**:	a one-dimensional string array containing the UTC spacecraft times in ISO8601 format
> ***Note***: this is the time at which the CCSDS packet was created on baord, contained in the packet data field header and converted from on-board time units to UTC on ground
+ **raw_val**: a one-dimensional string array containing the raw values of the parameters
+ **eng_val**: a one-dimensional string array containing the calibrated values of the parameters
> ***Note***: if calibration is not available for a parameter, the corresponding *eng_val* field has the value "N/A"
+ **unit**: a one-dimensional string array containing the physical units of the calibrated values
+ **desc**:	a one-dimensional string array containing descriptions of the parameter content as per instrument database definition
> ***Note***: for each house-keeping parameter, the JSON file must contain at least two values: the last value measured before the start of the observation which the data belongs to, and the first value measured after the end of the observation

### Output
The output data files generated by the pipeline are saved into the *output* folder. They are listed in a JSON file named *contents.json* having the following structure:

	{
	    "file_name": "",
	    "l0_file_name": "",
	    "log_file_name": "",
        "empty_params": [""]
	}

where all fields are defined as follows:

+ **file_name**: a string scalar containing the pathname of the generated L1 FITS file

> ***Note***: the FITS file is named according to the convention: `solo_L1_metis-[datatype]_[date-time]_V[version].fits`, where `[datatype]` identifies the Metis data product type, `[date-time]` is the 15-character date of observation, formatted as `yyyymmddThhmmss`, and `[version]` is the 2-digit, zero-padded incremental version number of the FITS file

+ **l0_file_name**: a string scalar containing the full pathname of the input L0 FITS file that has been processed
+ **log_file_name**: a string scalar containing the full pathname of an ASCII text file containing a log of the pipeline processing
+ **empty_params**: a string array containing the list of the house-keeping parameters for which raw values were used instead of calibrated values in generating the house-keeping extension of the output L1 file

## check_quality.pro
Makes a classification of the quality of the input image creating a matrix providing, pixel by pixel, a quality index.

### Call
	result = check_quality(image, cal_pack[, filter])

### Input
+ **image**: a two-dimensional 16-bit unsigned-integer array containing a raw Metis image
+ **cal_pack**: an IDL structure variable containing the applicable Metis calibration package

### Output
+ **result**: a two-dimensional floating-point array with same size as the input image containing the quality index values

> ***Note***: possible values of the quality index are: *NaN*, for all saturated pixel or pixels having a value well below the average detector bias level; 1 for all pixels whose values are within the detector linearity range; 0 for all remaining pixels.

### Optional input
+ **filter**: a string scalar indicating which Metis channel has acquired the input image. If not provided, the VL channel is assumed

## decode_obt.pro
See the documentation of the Metis L0 pipeline.

## fits_hdr2struct.pro
Converts a FITS header string array into an IDL data structure.

### Call
	result = fits_hdr2struct(string_header)

### Input
+ **string_header**: a one-dimensional string array containing a valid FITS header as obtained, e.g., from *fitsread*.*pro* or *mrdfits*.*pro*

### Output
+ **result**: an IDL structure variable containing the converted header

## get_light_time.pro
Computes the light time of a target body relative to an observer for the given date, and optionally returns its velocity vector.

### Call
	result = get_light_time(utc, target, observer[, rad_velocity = rad_velocity])
> ***Note***: a valid set of SPICE ephemeris and attitude kernels for the specified target body and observer must be loaded before calling this routine

### Input
+ **utc**: a string scalar containing the UTC date in ISO8601 format at which the light time is to be computed
+ **target**: a string scalar containing the name or SPICE ID code of the target body
+ **observer**: a string scalar containing the name or SPICE ID code of the observing body

### Output
+ **result**: a double-precision floating-point scalar with the one-way light time between the target body and the observer in seconds

### Optional output
+ **rad_velocity**: a named variable that will receive the double-precision floating-point scalar with the radial component of the target body velocity relative to the observer in km/s

## interpol_param.pro
Interpolates the table of Metis house-keeping parameters to get the value of a specified parameter at the given date. Calibrated values of the specified parameter are used by default for the interpolation; if they are not available in the input table, raw values are used instead.

### Call

    result = interpol_param(hk_table, par_name, date[, empty_params = empty_params])

### Input
+ **hk_table**: an IDL structure variable containing the Metis house-keeping parameter table provided as input to the L1 pipeline
+ **par_name**: a string scalar containing the name of the parameter for which interpolation is to be done
+ **date**: a string scalar containing the UTC date in ISO8601 format at which interpolation of the specified parameter is to be done

### Output
+ **result**: a floating-point scalar with the value resulting from the linear interpolation of the table values
> ***Note***: if the specified date is outside the range of dates for which values are given in the input table, or if no values are given for the specified parameter, a null value is returned

### Optional output
+ **empty_params**: a named string scalar or string array variable to which the string scalar with the name of the specified parameter is appended if calibrated values are not available for that parameter in the input table

## json_write.pro
See the documentation of the Metis L0 pipeline.

## load_spice_kernels.pro
Load or unload the Solar Orbiter's SPICE ephemeris and attitude kernels.

### Call

    load_spice_kernels, path[, unload = unload][, kernel_list = kernel_list][, kernel_version = kernel_version]

### Input
+ **path**: a string scalar containing the full path of the Solar Orbiter's SPICE kernels folder

### Output
None.

### Optional input
+ **unload**: set this keyword to unload the kernels
> ***Note***: if this keyword is set, the list of loaded kernels must be passed through the *kernel_list* argument

### Optional output
+ **kernel_list**: a named variable that will receive the string array with the full pathnames of the kernel files that have been loaded
+ **kernel_version**: a named variable that will receive the string scalar with the version identifier of the kernel dataset that has been loaded

## make_bin_table.pro
Returns an IDL structure variable containing a copy of the Metis house-keeping parameter table with the proper format to be saved as a binary table in an L1 FITS file extension.

### Call

    result = make_bin_table(hk_table)

### Input
+ **hk_table**: an IDL structure variable containing the Metis house-keeping parameter table provided as input to the L1 pipeline

### Output
+ **result**: an IDL structure variable containing the reformatted table

## metis_rectify.pro
Applies the proper transformation to bring the input image from Metis detector reference frame to the Sun-oriented detector reference frame

> ***Note***: the Sun-oriented detector reference frame is defined as the reference frame with both axes parallel to the detector sides, axis 1 (X) pointing towards the solar West, and axis 2 (Y) pointing to the solar North in the case the spacecraft roll is zero

### Call

    result = metis_rectify(image[, filter])

### Input
+ **image**: a two-dimensional array containing the detector-oriented Metis image

### Output:
+ **result**: a two-dimensional array with same size as the input image containing the Sun-oriented image

### Optional input
+ **filter**: a string scalar indicating which Metis channel has acquired the input image. If not provided, the VL channel is assumed

## metis_wcs.pro
Returns an IDL list variable containing the full set of FITS header keywords compliant with the World Coordinate System (WCS) standard. Information for the calculation of the WCS keywords are taken from the image FITS header, Solar Orbiter's SPICE attitude kernels, and applicable Metis calibration package.

### Call

    result = metis_wcs(header, cal_pack[, ref_detector = ref_detector])
> ***Note***: a valid set of Solar Orbiter's SPICE ephemeris and attitude kernels must be loaded before calling this routine

### Input
+ **header**: an IDL structure variable containing a valid Metis FITS header
> ***Note***: this routine will not work with a Metis L0 FITS header, since some relevant keyword values are computed only at level L1
+ **cal_pack**: an IDL structure variable containing the applicable Metis calibration package

### Output
+ **result**: an IDL list variable containing the full set of WCS keywords:
    
        WCSNAME             Name of coordinate system
        PCi_j               Transformation matrix (i, j = 1, 2)
        CRPIX1, CRPIX2      Reference pixel coordinates
        CDELT1, CDELT2      Pixel scales
        CRVAL1, CRVAL2      Values for reference pixel
        CTYPE1, CTYPE1      System projection types
        CUNIT1, CUNIT2      Units
        CROTA               Rotation angle (deprecated)
    plus other Metis-specific keywords

### Optional input
+ **ref_detector**: set this keyword to force the calculation of the WCS keywords for the detector-oriented image. The default is to compute them for the Sun-oriented image

## solo_get_carringrot.pro
Returns the decimal Carrington rotation number of Sun's disk center as seen by Solar Orbiter for the given date.

### Call

    result = solo_get_carringrot(utc)
> ***Note***: a valid set of Solar Orbiter's SPICE ephemeris and attitude kernels must be loaded before calling this routine

### Input
+ **utc**: a string scalar containing the UTC date in ISO8601 format at which the Carrington rotation number is to be computed

### Output
+ **result**: a floating-point scalar with the decimal Carrington rotation number

## solo_get_coords.pro
Returns the orbital state vector of Solar Orbiter in the specified reference frame relative to an observer for the given date.

### Call

    result = solo_get_coords(utc, frame, observer[, spherical = spherical][, degrees = degrees][, velocity = velocity])
> ***Note***: a valid set of Solar Orbiter's SPICE ephemeris and attitude kernels must be loaded before calling this routine

### Input
+ **utc**: a string scalar containing the UTC date in ISO8601 format at which the Solar Orbiter state is to be calculated
+ **frame**: a string scalar containing the name of one of the following standard coordinate systems:
        
        RTN             Radial-Tangential-Normal
        HEE             Heliocentric Earth Ecliptic
        HCI             Heliocentric Inertial
        HAE             Heliocentric Aries Ecliptic
        HEQ             Heliocentric Earth Equatorial
        GSE             Geocentric Solar Ecliptic
        GEI             Geocentric Equatorial Inertial
        CARRINGTON      Carrington system (can be abbreviated)
+ **observer**: a string scalar containing the name or SPICE ID code of the observing body

### Output
+ **result**: a double-precision floating-point array with the three Cartesian (default, in km) or spherical (either in radians or degrees) coordinates of Solar Orbiter in the specified reference frame

### Optional input
+ **spherical**: set this keyword to return spherical coordinates (distance in km, longitude and latitude in radians) instead of Cartesian coordinates
+ **degrees**: set this keyword to return longitude and latitude in degrees instead of radians

### Optional output
+ **velocity**: a named variable that will receive the double-precision floating-point array with the three Cartesian components of Solar Orbiter's velocity relative to the observer in km/s

## solo_get_ephemeris.pro

### Call
### Input
### Output