function metis_wcs, header, cal_pack, ref_detector = ref_detector

	if header.filter.contains('VL', /fold) then $
		channel = cal_pack.vl_channel else $
		channel = cal_pack.uv_channel

	boresight = channel.boresight[0]
	detector_size = channel.detector_size.value

	; compute correct boresight parameters to account for image orientation, binning, and fits convention

	; plate scale

	cdelt1 = boresight.plate_scale.value * header.nbin1
	cdelt2 = boresight.plate_scale.value * header.nbin2
	cdelt = [cdelt1, cdelt2]

	; old definitions if boresight parameters in pixel units are used

	; pntpix1 = boresight.pntpix1.value + (nominal_size - 1)/2.
	; pntpix1 = (pntpix1 + 0.5)/header.nbin1 + 0.5
	; pntpix2 = boresight.pntpix2.value + (nominal_size - 1)/2.
	; pntpix2 = (pntpix2 + 0.5)/header.nbin2 + 0.5

	; new definitions if boresight parameters in arcsec units are used

	detector_size = detector_size/sqrt(header.nbin)
	xcen = (detector_size + 1)/2.
	ycen = (detector_size + 1)/2.

	; boresight, i.e., io center

	borpix1 = boresight.borpix1.value/cdelt1 + xcen
	borpix2 = boresight.borpix2.value/cdelt2 + ycen
	borpix = [borpix1, borpix2]

	; s/c pointing

	pntpix1 = boresight.pntpix1.value/cdelt1 + xcen
	pntpix2 = boresight.pntpix2.value/cdelt2 + ycen
	pntpix = [pntpix1, pntpix2]

	; determine spacecract pointing information in the hpc reference frame using the spice kernels

	pointing = solo_get_pointing(header.date_avg, /degrees, /arcsec)

	; NOTE - values are defined as follows:
	; pointing[0] = yaw (arcsec)
	; pointing[1] = pitch (arcsec)
	; pointing[2] = roll (deg)

	; physical coordinates of s/c pointing (i.e., yaw and pitch) in the hpc reference frame

	pntval1 = pointing[0]
	pntval2 = pointing[1]
	pntval = [pntval1, pntval2]

	; correct the roll angle value for metis misalignment

	roll = (pointing[2] + boresight.delta_roll.value) * !dtor
	
	; wcs rotation matrix in the hpc reference frame

	pc = [[cos(roll), -sin(roll)], [sin(roll), cos(roll)]]

	ctype1 = 'HPLN-TAN'
	ctype2 = 'HPLT-TAN'

	; if requested, transform the wcs matrix to the detector reference frame and adjust the boresight and spacecraft pointing parameters

	if keyword_set(ref_detector) then begin
		if header.filter.contains('UV', /fold) then begin
			borpix_prime = borpix
			borpix[0] = detector_size - (borpix_prime[1] - 1.)
			borpix[1] = detector_size - (borpix_prime[0] - 1.)
			pntpix_prime = pntpix
			pntpix[0] = detector_size - (pntpix_prime[1] - 1.)
			pntpix[1] = detector_size - (pntpix_prime[0] - 1.)
			pc = -reverse(pc, 1)
			pc = transpose(pc)
		endif
		if header.filter.contains('VL', /fold) then begin
			borpix_prime = borpix
			borpix[0] = borpix_prime[1]
			borpix[1] = detector_size - (borpix_prime[0] - 1.)
			pntpix_prime = pntpix
			pntpix[0] = pntpix_prime[1]
			pntpix[1] = detector_size - (pntpix_prime[0] - 1.)
			roll = roll + !dpi/2.
			pc = [[cos(roll), -sin(roll)], [sin(roll), cos(roll)]]
		endif
	endif
	
	; get sun's center pixel

	sunval = [0., 0.]
	sunpix = (invert(pc) ## (sunval - pntval))/cdelt + pntpix

	; get coordinates of the image center pixel

	crpix = [xcen, ycen]
	crval = (pc ## (crpix - pntpix)) * cdelt + pntval

	; create the wcs list

	wcs = list()

	wcs.add, { $
		name: 'WCSNAME', $
		value: 'Helioprojective-Cartesian', $
		comment: 'name of coordinate system'}
	wcs.add, { $
		name: 'CTYPE1', $
		value: ctype1, $
		comment: ctype1 eq 'HPLT-TAN' ? 'helioprojective latitude (solar y)' : 'helioprojective longitude (solar x)'}
	wcs.add, { $
		name: 'CTYPE2', $
		value: ctype2, $
		comment: ctype2 eq 'HPLT-TAN' ? 'helioprojective latitude (solar y)' : 'helioprojective longitude (solar x)'}
	wcs.add, { $
		name: 'CUNIT1', $
		value: 'arcsec', $
		comment: 'units along axis 1'}
	wcs.add, { $
		name: 'CUNIT2', $
		value: 'arcsec', $
		comment: 'units along axis 2'}
	wcs.add, { $
		name: 'PC1_1', $
		value: pc[0, 0], $
		comment: 'WCS coordinate transformation matrix'}
	wcs.add, { $
		name: 'PC1_2', $
		value: pc[1, 0], $
		comment: 'WCS coordinate transformation matrix'}
	wcs.add, { $
		name: 'PC2_1', $
		value: pc[0, 1], $
		comment: 'WCS coordinate transformation matrix'}
	wcs.add, { $
		name: 'PC2_2', $
		value: pc[1, 1], $
		comment: 'WCS coordinate transformation matrix'}
	wcs.add, { $
		name: 'CDELT1', $
		value: cdelt[0], $
		comment: '[arcsec] pixel scale along axis 1'}
	wcs.add, { $
		name: 'CDELT2', $
		value: cdelt[1], $
		comment: '[arcsec] pixel scale along axis 2'}
	wcs.add, { $
		name: 'CROTA', $
		value: atan(pc[0, 1], pc[0, 0]) * !radeg, $
		comment: '[deg] rotation angle'}
	wcs.add, { $
		name: 'CRVAL1', $
		value: crval[0], $
		comment: '[arcsec] value of reference pixel along axis 1'}
	wcs.add, { $
		name: 'CRVAL2', $
		value: crval[1], $
		comment: '[arcsec] value of reference pixel along axis 2'}
	wcs.add, { $
		name: 'CRPIX1', $
		value: crpix[0], $
		comment: '[pixel] reference pixel location along axis 1'}
	wcs.add, { $
		name: 'CRPIX2', $
		value: crpix[1], $
		comment: '[pixel] reference pixel location along axis 2'}
	wcs.add, { $
		name: 'SUN_XCEN', $
		value: sunpix[0], $
		comment: '[pixel] sun center location along axis 1'}
	wcs.add, { $
		name: 'SUN_YCEN', $
		value: sunpix[1], $
		comment: '[pixel] sun center location along axis 2'}
	wcs.add, { $
		name: 'SUNPIX1', $
		value: sunpix[0], $
		comment: '[pixel] sun center location along axis 1'}
	wcs.add, { $
		name: 'SUNPIX2', $
		value: sunpix[1], $
		comment: '[pixel] sun center location along axis 2'}
	wcs.add, { $
		name: 'IO_XCEN', $
		value: borpix[0], $
		comment: '[pixel] Metis IO center location along axis 1'}
	wcs.add, { $
		name: 'IO_YCEN', $
		value: borpix[1], $
		comment: '[pixel] Metis IO center location along axis 2'}
	wcs.add, { $
		name: 'IOPIX1', $
		value: borpix[0], $
		comment: '[pixel] Metis IO center location along axis 1'}
	wcs.add, { $
		name: 'IOPIX2', $
		value: borpix[1], $
		comment: '[pixel] Metis IO center location along axis 2'}
	wcs.add, { $
		name: 'FS_XCEN', $
		value: crpix[0], $
		comment: '[pixel] Metis FS center location along axis 1'}
	wcs.add, { $
		name: 'FS_YCEN', $
		value: crpix[1], $
		comment: '[pixel] Metis FS center location along axis 2'}
	wcs.add, { $
		name: 'FSPIX1', $
		value: crpix[0], $
		comment: '[pixel] Metis FS center location along axis 1'}
	wcs.add, { $
		name: 'FSPIX2', $
		value: crpix[1], $
		comment: '[pixel] Metis FS center location along axis 2'}
	wcs.add, { $
		name: 'SC_XCEN', $
		value: pntpix[0], $
		comment: '[pixel] S/C pointing location along axis 1'}
	wcs.add, { $
		name: 'SC_YCEN', $
		value: pntpix[1], $
		comment: '[pixel] S/C pointing location along axis 2'}
	wcs.add, { $
		name: 'SCPIX1', $
		value: pntpix[0], $
		comment: '[pixel] S/C pointing location along axis 1'}
	wcs.add, { $
		name: 'SCPIX2', $
		value: pntpix[1], $
		comment: '[pixel] S/C pointing location along axis 2'}
	wcs.add, { $
		name: 'SC_YAW', $
		value: pointing[0], $
		comment: '[arcsec] S/C HPC yaw'}
	wcs.add, { $
		name: 'SC_PITCH', $
		value: pointing[1], $
		comment: '[arcsec] S/C HPC pitch'}
	wcs.add, { $
		name: 'SC_ROLL', $
		value: pointing[2], $
		comment: '[deg] S/C HPC roll angle'}
	wcs.add, { $
		name: 'INN_FOV', $
		value: 1.6, $
		comment: '[deg] inner Metis FOV'}
	wcs.add, { $
		name: 'OUT_FOV', $
		value: 3.4, $
		comment: '[deg] outer Metis FOV'}

	return, wcs
end
