function solo_get_ephemeris, header, cal_pack

	; ephemerides are to be calculated for the average date of observation

	utc = header.date_avg

	; get useful physical constant

	rsun = cal_pack.constants.rsun.value
	au = cal_pack.constants.au.value

	; compute and add the wcs keyword

	ephemerides = list()

	ephemerides.add, { $
		name: 'LONPOLE', $
		value: 180., $
		comment: '[deg] native longitude of the celestial pole'}

	; spherical coordinates of the S/C in the heeq (i.e., stonyhurst) frame

	coord = solo_get_coords(utc, 'HEQ', 'SUN', /spherical, /degrees)

	; solar photospheric radius as seen from the S/C

	solo_dist = coord[0] * 1000.
	rsun_arc = atan(rsun, solo_dist) * !radeg * 3600.d0

	ephemerides.add, { $
		name: 'RSUN_ARC', $
		value: rsun_arc, $
		comment: '[arcsec] apparent photospheric solar radius'}
	ephemerides.add, { $
		name: 'RSUN_REF', $
		value: rsun, $
		comment: '[m] assumed physical solar radius'}

	solar_angles = solo_get_solar_angles(utc)

	ephemerides.add, { $
		name: 'SOLAR_B0', $
		value: solar_angles[0], $
		comment: '[deg] S/C tilt of solar north pole'}
	ephemerides.add, { $
		name: 'SOLAR_P0 ', $
		value: solar_angles[1], $
		comment: '[deg] S/C celestial north to solar north angle'}
	ephemerides.add, { $
		name: 'SOLAR_EP', $
		value: solar_angles[2], $
		comment: '[deg] S/C ecliptic north to solar north angle'}

	carrot = solo_get_carringrot(utc)

	ephemerides.add, { $
		name: 'CAR_ROT', $
		value: carrot, $
		comment: 'carrington rotation number'}

	ephemerides.add, { $
		name: 'HGLT_OBS', $
		value: coord[2], $
		comment: '[deg] S/C heliographic latitude (B0 angle)'}
	ephemerides.add, { $
		name: 'HGLN_OBS', $
		value: coord[1], $
		comment: '[deg] S/C heliographic longitude'}

	; coordinates of the S/C in the carrington frame and carrington rotation number

	coord = solo_get_coords(utc, 'CARRINGTON', 'SUN', /spherical, /degrees)

	ephemerides.add, { $
		name: 'CRLT_OBS', $
		value: coord[2], $
		comment: '[deg] S/C carrington latitude (B0 angle)'}
	ephemerides.add, { $
		name: 'CRLN_OBS', $
		value: coord[1], $
		comment: '[deg] S/C carrington longitude (L0 angle)'}

	ephemerides.add, { $
		name: 'DSUN_OBS', $
		value: solo_dist, $
		comment: '[m] S/C distance from sun'}
	ephemerides.add, { $
		name: 'DSUN_AU', $
		value: solo_dist/au, $
		comment: '[AU] S/C distance from sun'}
	ephemerides.add, { $
		name: 'AU_REF', $
		value: au, $
		comment: '[m] assumed physical astronomical unit'}

	; coordinates of the S/C in the hee frame

	coord = solo_get_coords(utc, 'HEE', 'SUN')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'HEEX_OBS', $
		value: coord[0], $
		comment: '[m] S/C heliocentric earth ecliptic x'}
	ephemerides.add, { $
		name: 'HEEY_OBS', $
		value: coord[1], $
		comment: '[m] S/C heliocentric earth ecliptic y'}
	ephemerides.add, { $
		name: 'HEEZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C heliocentric earth ecliptic z'}

	; coordinates of the S/C in the hci frame

	coord = solo_get_coords(utc, 'HCI', 'SUN', vel = vel)
	coord = coord * 1000.
	vel = vel * 1000.

	ephemerides.add, { $
		name: 'HCIX_OBS', $
		value: coord[0], $
		comment: '[m] S/C heliocentric inertial x'}
	ephemerides.add, { $
		name: 'HCIY_OBS', $
		value: coord[1], $
		comment: '[m] S/C heliocentric inertial y'}
	ephemerides.add, { $
		name: 'HCIZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C heliocentric inertial z'}

	ephemerides.add, { $
		name: 'HCIX_VOB', $
		value: vel[0], $
		comment: '[m/s] S/C heliocentric inertial x velocity'}
	ephemerides.add, { $
		name: 'HCIY_VOB', $
		value: vel[1], $
		comment: '[m/s] S/C heliocentric inertial y velocity'}
	ephemerides.add, { $
		name: 'HCIZ_VOB', $
		value: vel[2], $
		comment: '[m/s] S/C heliocentric inertial z velocity'}

	; coordinates of the S/C in the hae frame

	coord = solo_get_coords(utc, 'HAE', 'SUN')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'HAEX_OBS', $
		value: coord[0], $
		comment: '[m] S/C heliocentric aries ecliptic x'}
	ephemerides.add, { $
		name: 'HAEY_OBS', $
		value: coord[1], $
		comment: '[m] S/C heliocentric aries ecliptic y'}
	ephemerides.add, { $
		name: 'HAEZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C heliocentric aries ecliptic z'}

	; coordinates of the S/C in the heeq frame

	coord = solo_get_coords(utc, 'HEQ', 'SUN')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'HEQX_OBS', $
		value: coord[0], $
		comment: '[m] S/C heliocentric earth equatorial x'}
	ephemerides.add, { $
		name: 'HEQY_OBS', $
		value: coord[1], $
		comment: '[m] S/C heliocentric earth equatorial y'}
	ephemerides.add, { $
		name: 'HEQZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C heliocentric earth equatorial z'}

	; coordinates of the S/C in the gse frame

	coord = solo_get_coords(utc, 'GSE', 'EARTH')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'GSEX_OBS', $
		value: coord[0], $
		comment: '[m] S/C geocentric solar ecliptic x'}
	ephemerides.add, { $
		name: 'GSEY_OBS', $
		value: coord[1], $
		comment: '[m] S/C geocentric solar ecliptic y'}
	ephemerides.add, { $
		name: 'GSEZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C geocentric solar ecliptic z'}

	; light travel times and radial velocity of the S/C

	earth_time = get_light_time(utc, 'EARTH', 'SUN')
	sun_time = get_light_time(utc, 'SOLO', 'SUN', rad_vel = rad_vel)
	t_del = earth_time - sun_time

	ephemerides.add, { $
		name: 'OBS_VR', $
		value: rad_vel, $
		comment: '[m/s] radial velocity of S/C relative to sun'}
	ephemerides.add, { $
		name: 'EAR_TDEL', $
		value: t_del, $
		comment: '[s] time(sun to earth) - time(sun to S/C)'}
	ephemerides.add, { $
		name: 'SUN_TIME', $
		value: sun_time, $
		comment: '[s] time(sun to S/C)'}

	; corrections of the acquisition date

	utc = header.date_beg

	jul_utc = date_conv(utc, 'julian')
	date_ear = date_conv(jul_utc + t_del/86400., 'fits')
	date_sun = date_conv(jul_utc - sun_time/86400., 'fits')

	ephemerides.add, { $
		name: 'DATE_EAR', $
		value: date_ear, $
		comment: '[UTC] obs. start time corrected to earth'}
	ephemerides.add, { $
		name: 'DATE_SUN', $
		value: date_sun, $
		comment: '[UTC] obs. start time corrected to sun'}

	return, ephemerides
end
