pro metis_l2_prep_vl_generic

	; keyword defining if the detector reference frame must be used for the output

	ref_detector = 1

	; start the log

	journal,'output/metis_l2_prep_log.txt'

	; read the auxiliary file - here we have all the inputs we need

	input = json_parse('input/contents.json', /toarray, /tostruct)

	; load the spice kernels

	load_spice_kernels, input.spice_kernels, kernel_list = kernel_list, kernel_version = kernel_version

	; read the calibration package

	cal_pack = json_parse(input.cal_pack_path + '/index.json', /toarray, /tostruct)

	; include the calibration package path into the cal_pack structure

	cal_pack = create_struct('path', input.cal_pack_path + path_sep(), cal_pack)

	; read the primary hdu

	data = mrdfits(input.file_name, 0, primary_header, /silent)

	; read the quality matrix

	quality_matrix = mrdfits(input.file_name, 'quality matrix', /silent)

	; convert the string header into an idl structure

	header = fits_hdr2struct(primary_header)

	; read the house-keeping extension

	hk_table = mrdfits(input.file_name, 'house-keeping', hk_extension_header, /silent)

	; calibration block

	if header.datatype eq 0 then begin
		data = double(data)

		; ====================================
		; for old l1 data
		; data = data * header.nbin * header.ndit
		; header.xposure = header.xposure * header.ndit
		; header.nsumexp = header.ndit
		; ====================================

		data = metis_dark_vlda(data, header, cal_pack, history = history)

		; ====================================
		; for data already subtracted of dark
		; file = file_basename(header.parent, '.fits') + '_subdark.fits'
		; data = mrdfits('VL subdark/' + file, 0, /silent)
		; data = rebin(data, header.naxis1, header.naxis2)
		; data = data * header.nbin * header.ndit
		; history = ['Dark correction: ', '  ' + file]
		; ====================================

		data = metis_flat_field(data, header, cal_pack, history = history)
		data = metis_vignetting(data, header, cal_pack, history = history)

		if header.pol_id ge 1 and header.pol_id le 4 then polarimetric = 1 else polarimetric = 0
		data = metis_rad_cal(data, header, cal_pack, polarimetric = polarimetric, history = history)

		if not ref_detector then data = rotate(data, 1)
	endif

	if header.datatype eq 3 then begin
		; calibration of temporal noise images
	endif

	if header.datatype eq 5 then begin
		; calibration of cr/sep log matrices
	endif

	; definitions for the primary header
	; version of the fits file

	version = string(input.l2_version + 1, format = '(I02)')

	; fits creation date

	date = date_conv(systime(/julian, /utc), 'FITS')

	; name of the fits file

	file_name_fields = strsplit(header.filename, '_', /extract)
	file_name = 'solo_L2_' + file_name_fields[2] + '_' + file_name_fields[3] + '_V' + version + '.fits'
	out_file_name = 'output/' + file_name

	; adjust the primary header

	fxaddpar, primary_header, 'FILENAME', file_name
	fxaddpar, primary_header, 'PARENT', file_basename(input.file_name)
	fxaddpar, primary_header, 'LEVEL', 'L2-draft'
	fxaddpar, primary_header, 'ORIGIN', ''
	fxaddpar, primary_header, 'CREATOR', 'metis_l2_prep_vl_generic.pro'
	fxaddpar, primary_header, 'VERS_SW', input.sw_version
	fxaddpar, primary_header, 'VERS_CAL', cal_pack.version, after = 'VERS_SW'
	case header.pol_id of
		0: fxaddpar, primary_header, 'BTYPE', 'VL fixed-polarization intensity'
		1: fxaddpar, primary_header, 'BTYPE', 'VL fixed-polarization intensity'
		2: fxaddpar, primary_header, 'BTYPE', 'VL fixed-polarization intensity'
		3: fxaddpar, primary_header, 'BTYPE', 'VL fixed-polarization intensity'
		4: fxaddpar, primary_header, 'BTYPE', 'VL fixed-polarization intensity'
		5: fxaddpar, primary_header, 'BTYPE', 'VL total brightness'
		6: fxaddpar, primary_header, 'BTYPE', 'VL total brightness'
	endcase
	fxaddpar, primary_header, 'BUNIT', cal_pack.vl_channel.cal_units
	fxaddpar, primary_header, 'DATAMIN', min(data, /nan)
	fxaddpar, primary_header, 'DATAMAX', max(data, /nan)
	fxaddpar, primary_header, 'XPOSURE', header.xposure
	fxaddpar, primary_header, 'NSUMEXP', header.nsumexp

	; append wcs keywords

	wcs = metis_wcs(header, cal_pack, ref_detector = ref_detector)

	foreach element, wcs do fxaddpar, primary_header, element.name, element.value, element.comment, before = 'DATATYPE'

	; append solar ephemeris keywords

	ephemerides = solo_get_ephemerides(header, constants = cal_pack.constants)
	foreach element, ephemerides do fxaddpar, primary_header, element.name, element.value, element.comment, before = 'DATATYPE'

	history = [history, 'Solar ephemerides and WCS:', '  SKD version = ' + kernel_version]

	; delete useless keywords

	sxdelpar, primary_header, 'BLANK'

	; add the history keyword

	for k = 0, n_elements(history) - 1 do fxaddpar, primary_header, 'HISTORY', history[k]

	fxaddpar, primary_header, 'HISTORY', 'L2 FITS file created on ' + date

	; add checksum and datasum to the fits header

	data = float(data)
	fits_add_checksum, primary_header, data
	mwrfits, data, out_file_name, primary_header, /no_comment, /create, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Quality matrix', 'Extension name'
	fits_add_checksum, extension_header, quality_matrix
	mwrfits, quality_matrix, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Error matrix', 'Extension name'
	fits_add_checksum, extension_header, intarr(header.naxis1, header.naxis2)
	mwrfits, intarr(header.naxis1, header.naxis2), out_file_name, extension_header, /no_comment, /silent

	; write the auxiliary information file

	output = { $
		file_name: out_file_name, $
		l2_file_name: input.file_name, $
		log_file_name: 'output/metis_l2_prep_log.txt' $
	}

	json_write, output, 'output/contents.json'

	; unload the spice kernels

	load_spice_kernels, kernel_list = kernel_list, /unload

	; close the log

	journal
	exit, status = 0

	error_handling:
	journal, 'Errors occurred while processing.', /continue
	journal
	exit, status = 1
end
