pro metis_l2_prep_vl_polariz

	; keyword defining if the detector reference frame must be used for the output

	ref_detector = 1

	; start the log

	journal,'output/metis_l2_prep_log.txt'

	; read the auxiliary file - here we have all the inputs we need

	input = json_parse('input/contents.json', /toarray, /tostruct)

	; load the spice kernels

	load_spice_kernels, input.spice_kernels, kernel_list = kernel_list, kernel_version = kernel_version

	; read the calibration package

	cal_pack = json_parse(input.cal_pack_path + '/index.json', /toarray, /tostruct)

	; include the calibration package path into the cal_pack structure

	cal_pack = create_struct('path', input.cal_pack_path + path_sep(), cal_pack)

	n = n_elements(input.file_name)

	; check on the number of input images (= npol = 4)

	if n ne 4 then begin
		error_message = 'number of input images is not equal to 4'
		goto, error_handling
	endif

	; calibration block

	data = !null
	data_header = !null
	data_subdark = !null

	quality_matrix = 1.

	for k = 0, 3 do begin

		; read the input image

		image = mrdfits(input.file_name[k], 0, primary_header, /silent)

		header = fits_hdr2struct(primary_header)

		; ====================================
		; for old l1 data
		; image = image * header.nbin * header.ndit
		; header.xposure = header.dit/1000. * header.ndit
		; header.nsumexp = header.ndit
		; ====================================

		; check consistency of polarization state

		if header.pol_id lt 1 or header.pol_id gt 4 then begin
			error_message = 'image no. ' + string(k, format = '(I0)') + $
			' has inconsistent polarization state'
			goto, error_handling
		endif

		; check consistency of pmp raw voltages (dacpol)

		if $
			header.dac1pol1 ne header.dac2pol1 or $
			header.dac1pol2 ne header.dac2pol2 or $
			header.dac1pol3 ne header.dac2pol3 or $
			header.dac1pol4 ne header.dac2pol4 then begin
			error_message = 'image no. ' + string(k, format = '(I0)') + ' has inconsistent PMP voltages'
			goto, error_handling
		endif

		; pile up images and headers

		data = [[[data]], [[image]]]
		data_header = [data_header, header]

		; read and update the quality matrix

		quality_matrix = quality_matrix * mrdfits(input.file_name[k], 'quality matrix', /silent)

		; apply dark correction to compute stokes i and total brightness

		tb_history = !null
		image_subdark = metis_dark_vlda(image, header, cal_pack, history = tb_history)

		; ====================================
		; for data already subtracted of dark
		; file = file_basename(header.parent, '.fits') + '_subdark.fits'
		; image_subdark = mrdfits('VL subdark/' + file, 0, /silent)
		; image_subdark = rebin(image_subdark, header.naxis1, header.naxis2)
		; image_subdark = image_subdark * header.nbin * header.ndit
		; tb_history = ['Dark correction: ', '  ' + file]
		; ====================================

		data_subdark = [[[data_subdark]], [[image_subdark]]]
	endfor

	; adjust header keywords

	obt_beg = min(data_header.obt_beg, i)
	obt_end = max(data_header.obt_end, j)

	header = data_header[0]
	header.date_beg = data_header[i].date_beg
	header.date_obs = data_header[i].date_beg
	header.date_end = data_header[j].date_end

	telapse = obt_end - obt_beg
	obt_avg = (obt_beg + obt_end)/2.
	date_avg = solo_obt2utc(decode_obt(obt_avg, /from_decimal))

	header.obt_beg = obt_beg
	header.obt_end = obt_end
	header.date_avg = date_avg

	header.tsensor = mean(data_header.tsensor)
	header.pmptemp = mean(data_header.pmptemp)

	; read the demodulation object of the calibration package

	demod_info = cal_pack.vl_channel.demodulation

	; read the calibration curve to convert pmp raw voltages (dacpol) into effective polarization angles

	dacpol_cal = cal_pack.vl_channel.dacpol_cal

	; apply the demodulation

	demod_tensor = fltarr(header.naxis1, header.naxis2, 4)
	stokes = dblarr(header.naxis1, header.naxis2, 3)
	stokes_subdark = dblarr(header.naxis1, header.naxis2, 3)
	stokes_name = ['I', 'Q', 'U']
	for i = 0, 2 do begin
		for j = 0, 3 do begin

			; check the polarization state of the image and select the corresponding dacpol value

			case data_header[j].pol_id of
				1: dacpol = data_header[j].dac1pol1
				2: dacpol = data_header[j].dac1pol2
				3: dacpol = data_header[j].dac1pol3
				4: dacpol = data_header[j].dac1pol4
			endcase

			; select the correct demodulation tensor element based on effective angle and stokes paramater

			k = where(dacpol_cal.dacpol eq dacpol)
			angle = dacpol_cal.angle[k]

			n = where(demod_info.angle eq angle[0] and demod_info.stokes eq stokes_name[i])
			demod_file = demod_info[n].file_name

			demod_image = float(readfits(cal_pack.path + demod_file, /silent))

			; rebin the demodulation tensor image to match image size

			demod_image = rebin(demod_image, header.naxis1, header.naxis2)
			demod_tensor[*, *, j] = demod_image
		endfor

		; compute the stokes parameters

		stokes[*, *, i] = $
			demod_tensor[*, *, 0] * data[*, *, 0] + $
			demod_tensor[*, *, 1] * data[*, *, 1] + $
			demod_tensor[*, *, 2] * data[*, *, 2] + $
			demod_tensor[*, *, 3] * data[*, *, 3]

		; compute the dark-subtracted stokes parameters

		stokes_subdark[*, *, i] = $
			demod_tensor[*, *, 0] * data_subdark[*, *, 0] + $
			demod_tensor[*, *, 1] * data_subdark[*, *, 1] + $
			demod_tensor[*, *, 2] * data_subdark[*, *, 2] + $
			demod_tensor[*, *, 3] * data_subdark[*, *, 3]
	endfor

	; compute the tb from the dark-subtracted stokes i and apply other calibrations

	tb_image = reform(stokes_subdark[*, *, 0])
	tb_image = metis_flat_field(tb_image, header, cal_pack, history = tb_history)
	tb_image = metis_vignetting(tb_image, header, cal_pack, history = tb_history)
	tb_image = metis_rad_cal(tb_image, header, cal_pack, /polarimetric, history = tb_history)

	if not ref_detector then tb_image = rotate(tb_image, 1)

	; compute the pb from the stokes q and u and apply other calibrations

	pb_image = sqrt(reform(stokes[*, *, 1])^2 + reform(stokes[*, *, 2])^2)
	pb_image = metis_flat_field(pb_image, header, cal_pack, history = pb_history)
	pb_image = metis_vignetting(pb_image, header, cal_pack, history = pb_history)
	pb_image = metis_rad_cal(pb_image, header, cal_pack, /polarimetric, history = pb_history)

	if not ref_detector then pb_image = rotate(pb_image, 1)

	; compute the polarization angle from the stokes q and u

	pol_angle = 0.5d0 * atan(stokes[*, *, 2], stokes[*, *, 1]) * !radeg

	if not ref_detector then pol_angle = rotate(pol_angle, 1)

	; definitions for the primary header
	; version of the fits file

	version = string(input.l2_version + 1, format = '(I02)')

	; creation and acquisition times in utc

	date = date_conv(systime(/julian, /utc), 'FITS')

	; adjust the primary header

	fxaddpar, primary_header, 'PARENT', strjoin(file_basename(input.file_name), ', ')
	fxaddpar, primary_header, 'LEVEL', 'L2-draft'
	fxaddpar, primary_header, 'ORIGIN', ''
	fxaddpar, primary_header, 'CREATOR', 'metis_l2_prep_vl_polariz.pro'
	fxaddpar, primary_header, 'VERS_SW', input.sw_version
	fxaddpar, primary_header, 'VERS_CAL', cal_pack.version, after = 'VERS_SW'
	fxaddpar, primary_header, 'DATE', date, 'Date and time of FITS file creation'
	fxaddpar, primary_header, 'DATE-BEG', header.date_beg, 'Start time of observation'
	fxaddpar, primary_header, 'DATE-OBS', header.date_obs, 'Same as DATE-BEG'
	fxaddpar, primary_header, 'DATE-AVG', header.date_avg, 'Average time of observation'
	fxaddpar, primary_header, 'DATE-END', header.date_end, 'End time of observation'
	fxaddpar, primary_header, 'OBT_BEG', header.obt_beg, 'Start acquisition time in on-board time', format = 'F0.5'
	fxaddpar, primary_header, 'OBT_END', header.obt_end, 'End acquisition time in on-board time', format = 'F0.5'
	fxaddpar, primary_header, 'TELAPSE', telapse, '[s] Elapsed time between beginning and end of observation'
	fxaddpar, primary_header, 'XPOSURE', header.xposure
	fxaddpar, primary_header, 'NSUMEXP', header.nsumexp
	fxaddpar, primary_header, 'TSENSOR', header.tsensor
	fxaddpar, primary_header, 'PMPTEMP', header.pmptemp

	; append wcs keywords

	wcs = metis_wcs(header, cal_pack, ref_detector = ref_detector)

	foreach element, wcs do fxaddpar, primary_header, element.name, element.value, element.comment, before = 'DATATYPE'

	; append solar ephemeris keywords

	ephemerides = solo_get_ephemerides(header, constants = cal_pack.constants)
	foreach element, ephemerides do fxaddpar, primary_header, element.name, element.value, element.comment, before = 'DATATYPE'

	history = ['Solar ephemerides and WCS:', '  SKD version = ' + kernel_version]
	tb_history = [tb_history, history]
	pb_history = [pb_history, history]

	; delete useless keywords

	sxdelpar, primary_header, 'OBJ_CNT'
	sxdelpar, primary_header, 'POL_ID'
	sxdelpar, primary_header, 'SNRMIN'
	sxdelpar, primary_header, 'SNRMAX'
	sxdelpar, primary_header, 'NB_IMG'
	sxdelpar, primary_header, 'SN_MEAN1'
	sxdelpar, primary_header, 'SN_VAR1'
	sxdelpar, primary_header, 'SN_MEAN2'
	sxdelpar, primary_header, 'SN_VAR2'
	sxdelpar, primary_header, 'SN_MEAN3'
	sxdelpar, primary_header, 'SN_VAR3'
	sxdelpar, primary_header, 'SN_MEAN4'
	sxdelpar, primary_header, 'SN_VAR4'
	sxdelpar, primary_header, 'SN_MEAN5'
	sxdelpar, primary_header, 'SN_VAR5'
	sxdelpar, primary_header, 'N'

	date_beg_string = strmid(header.date_beg, 0, 19)
	date_beg_string = date_beg_string.replace('-', '')
	date_beg_string = date_beg_string.replace(':', '')

	; keywords specific for polarized brightness images

	primary_pb_header = primary_header

	; name of the fits file

	file_name = 'solo_L2_metis-vl-pb_' + date_beg_string + '_V' + version + '.fits'
	out_file_name = 'output/' + file_name

	fxaddpar, primary_pb_header, 'FILENAME', file_name
	fxaddpar, primary_pb_header, 'BTYPE', 'VL polarized brightness'
	fxaddpar, primary_pb_header, 'BUNIT', cal_pack.vl_channel.cal_units
	fxaddpar, primary_pb_header, 'DATAMIN', min(pb_image, /nan)
	fxaddpar, primary_pb_header, 'DATAMAX', max(pb_image, /nan)
	sxdelpar, primary_pb_header, 'BLANK'

	; add the history keyword

	for k = 0, n_elements(pb_history) - 1 do fxaddpar, primary_pb_header, 'HISTORY', pb_history[k]

	fxaddpar, primary_pb_header, 'HISTORY', 'L2 FITS file created on ' + date

	; add checksum and datasum to the fits header

	fits_add_checksum, primary_pb_header, pb_image

	mwrfits, pb_image, out_file_name, primary_pb_header, /no_comment, /create, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Quality matrix', 'Extension name'
	fits_add_checksum, extension_header, quality_matrix
	mwrfits, quality_matrix, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Error matrix', 'Extension name'
	fits_add_checksum, extension_header, intarr(header.naxis1, header.naxis2)
	mwrfits, intarr(header.naxis1, header.naxis2), out_file_name, extension_header, /no_comment, /silent

	; keywords specific for total brightness images

	primary_tb_header = primary_header

	; name of the fits file

	file_name = 'solo_L2_metis-vl-tb_' + date_beg_string + '_V' + version + '.fits'
	out_file_name = 'output/' + file_name

	fxaddpar, primary_tb_header, 'FILENAME', file_name
	fxaddpar, primary_tb_header, 'BTYPE', 'VL total brightness'
	fxaddpar, primary_tb_header, 'BUNIT', cal_pack.vl_channel.cal_units
	fxaddpar, primary_tb_header, 'DATAMIN', min(tb_image, /nan)
	fxaddpar, primary_tb_header, 'DATAMAX', max(tb_image, /nan)
	sxdelpar, primary_tb_header, 'BLANK'

	; add the history keyword

	for k = 0, n_elements(tb_history) - 1 do fxaddpar, primary_tb_header, 'HISTORY', tb_history[k]

	fxaddpar, primary_tb_header, 'HISTORY', 'L2 FITS file created on ' + date

	; add checksum and datasum to the fits header

	fits_add_checksum, primary_tb_header, tb_image

	mwrfits, tb_image, out_file_name, primary_tb_header, /no_comment, /create, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Quality matrix', 'Extension name'
	fits_add_checksum, extension_header, quality_matrix
	mwrfits, quality_matrix, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Error matrix', 'Extension name'
	fits_add_checksum, extension_header, intarr(header.naxis1, header.naxis2)
	mwrfits, intarr(header.naxis1, header.naxis2), out_file_name, extension_header, /no_comment, /silent

	; keywords specific for polarization-angle images

	primary_polangle_header = primary_header

	; name of the fits file

	file_name = 'solo_L2_metis-vl-pol-angle_' + date_beg_string + '_V' + version + '.fits'
	out_file_name = 'output/' + file_name

	fxaddpar, primary_polangle_header, 'FILENAME', file_name
	fxaddpar, primary_polangle_header, 'BTYPE', 'VL polarization angle'
	fxaddpar, primary_polangle_header, 'BUNIT', 'deg'
	fxaddpar, primary_polangle_header, 'DATAMIN', min(pol_angle, /nan)
	fxaddpar, primary_polangle_header, 'DATAMAX', max(pol_angle, /nan)
	sxdelpar, primary_polangle_header, 'BLANK'

	; add the history keyword

	for k = 0, n_elements(pb_history) - 1 do fxaddpar, primary_polangle_header, 'HISTORY', pb_history[k]

	fxaddpar, primary_polangle_header, 'HISTORY', 'L2 FITS file created on ' + date

	; add checksum and datasum to the fits header

	fits_add_checksum, primary_polangle_header, pol_angle

	mwrfits, pol_angle, out_file_name, primary_polangle_header, /no_comment, /create, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Quality matrix', 'Extension name'
	fits_add_checksum, extension_header, quality_matrix
	mwrfits, quality_matrix, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Error matrix', 'Extension name'
	fits_add_checksum, extension_header, intarr(header.naxis1, header.naxis2)
	mwrfits, intarr(header.naxis1, header.naxis2), out_file_name, extension_header, /no_comment, /silent

	; =================== STOKES PARAMETERS ======================

	i = reform(stokes_subdark[*, *, 0])
	i = metis_flat_field(i, header, cal_pack)
	i = metis_vignetting(i, header, cal_pack)
	i = metis_rad_cal(i, header, cal_pack, /polarimetric)
	if not ref_detector then i = rotate(i, 1)

	q = reform(stokes[*, *, 1])
	q = metis_flat_field(q, header, cal_pack)
	q = metis_vignetting(q, header, cal_pack)
	q = metis_rad_cal(q, header, cal_pack, /polarimetric)
	if not ref_detector then q = rotate(q, 1)

	u = reform(stokes[*, *, 2])
	u = metis_flat_field(u, header, cal_pack)
	u = metis_vignetting(u, header, cal_pack)
	u = metis_rad_cal(u, header, cal_pack, /polarimetric)
	if not ref_detector then u = rotate(u, 1)

	; keywords specific for stokes images

	primary_stokes_header = primary_header

	; name of the fits file

	file_name = 'solo_L2_metis-vl-stokes_' + date_beg_string + '_V' + version + '.fits'
	out_file_name = 'output/' + file_name

	fxaddpar, primary_stokes_header, 'FILENAME', file_name
	fxaddpar, primary_stokes_header, 'BTYPE', 'Stokes I'
	fxaddpar, primary_stokes_header, 'BUNIT', cal_pack.vl_channel.cal_units
	fxaddpar, primary_stokes_header, 'DATAMIN', min(i, /nan)
	fxaddpar, primary_stokes_header, 'DATAMAX', max(i, /nan)
	sxdelpar, primary_stokes_header, 'BLANK'

	; add the history keyword

	for k = 0, n_elements(tb_history) - 1 do fxaddpar, primary_stokes_header, 'HISTORY', tb_history[k]

	fxaddpar, primary_stokes_header, 'HISTORY', 'L2 FITS file created on ' + date

	; add checksum and datasum to the fits header

	fits_add_checksum, primary_stokes_header, i
	mwrfits, i, out_file_name, primary_stokes_header, /no_comment, /create, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Stokes Q', 'Extension name'
	fxaddpar, extension_header, 'BTYPE', 'Stokes Q'
	fxaddpar, extension_header, 'BUNIT', cal_pack.vl_channel.cal_units
	fxaddpar, extension_header, 'DATAMIN', min(q, /nan)
	fxaddpar, extension_header, 'DATAMAX', max(q, /nan)
	fits_add_checksum, extension_header, q
	mwrfits, q, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Stokes U', 'Extension name'
	fxaddpar, extension_header, 'BTYPE', 'Stokes U'
	fxaddpar, extension_header, 'BUNIT', cal_pack.vl_channel.cal_units
	fxaddpar, extension_header, 'DATAMIN', min(u, /nan)
	fxaddpar, extension_header, 'DATAMAX', max(u, /nan)
	fits_add_checksum, extension_header, u
	mwrfits, u, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'PCOUNT', 0, 'Parameter count'
	fxaddpar, extension_header, 'GCOUNT', 1, 'Group count'
	fxaddpar, extension_header, 'EXTNAME', 'Quality matrix', 'Extension name'
	fits_add_checksum, extension_header, quality_matrix
	mwrfits, quality_matrix, out_file_name, extension_header, /no_comment, /silent

	extension_header = !null
	fxaddpar, extension_header, 'EXTNAME', 'Error matrix', 'Extension name'
	mwrfits, intarr(header.naxis1, header.naxis2), out_file_name, extension_header, /no_comment, /silent

	; write the auxiliary information file

	output = { $
		file_name: out_file_name, $
		l2_file_name: input.file_name, $
		log_file_name: 'output/metis_l2_prep_log.txt' $
	}

	json_write, output, 'output/contents.json'

	; unload the spice kernels

	load_spice_kernels, kernel_list = kernel_list, /unload

	; close the log

	journal
	;exit, status = 0
	return

	error_handling:
	journal, 'Errors occurred while processing:'
	journal, '  ' + error_message
	journal
	exit, status = 1
end
