function solo_get_ephemerides, header, constants = constants

	utc = header.date_avg

	rsun = constants.rsun.value
	au = constants.au.value

	; compute and add the wcs keyword

	ephemerides = list()

	ephemerides.add, { $
		name: 'LONPOLE', $
		value: 180., $
		comment: '[deg] Native longitude of the celestial pole'}

	; spherical coordinates of the S/C in the heeq (i.e., stonyhurst) frame

	coord = solo_get_coords(utc, 'HEQ', 'SUN', /spherical, /degrees)

	; solar photospheric radius as seen from the S/C

	solo_dist = coord[0] * 1000.
	rsun_arc = atan(rsun, solo_dist) * !radeg * 3600.d0

	ephemerides.add, { $
		name: 'RSUN_ARC', $
		value: rsun_arc, $
		comment: '[arcsec] Apparent photospheric solar radius'}
	ephemerides.add, { $
		name: 'RSUN_REF', $
		value: rsun, $
		comment: '[m] Assumed physical solar radius'}

	solar_angles = solo_get_solar_angles(utc)

	ephemerides.add, { $
		name: 'SOLAR_B0', $
		value: solar_angles[0], $
		comment: '[deg] S/C tilt of solar North pole'}
	ephemerides.add, { $
		name: 'SOLAR_P0 ', $
		value: solar_angles[1], $
		comment: '[deg] S/C celestial North to solar North angle'}
	ephemerides.add, { $
		name: 'SOLAR_EP', $
		value: solar_angles[2], $
		comment: '[deg] S/C ecliptic North to solar North angle'}

	carrot = solo_get_carrot(utc)

	ephemerides.add, { $
		name: 'CAR_ROT', $
		value: carrot, $
		comment: 'Carrington rotation number'}

	ephemerides.add, { $
		name: 'HGLT_OBS', $
		value: coord[2], $
		comment: '[deg] S/C heliographic latitude (B0 angle)'}
	ephemerides.add, { $
		name: 'HGLN_OBS', $
		value: coord[1], $
		comment: '[deg] S/C heliographic longitude'}

	; coordinates of the S/C in the carrington frame and carrington rotation number

	coord = solo_get_coords(utc, 'CARRINGTON', 'SUN', /spherical, /degrees)

	ephemerides.add, { $
		name: 'CRLT_OBS', $
		value: coord[2], $
		comment: '[deg] S/C Carrington latitude (B0 angle)'}
	ephemerides.add, { $
		name: 'CRLN_OBS', $
		value: coord[1], $
		comment: '[deg] S/C Carrington longitude (L0 angle)'}

	ephemerides.add, { $
		name: 'DSUN_OBS', $
		value: solo_dist, $
		comment: '[m] S/C distance from Sun'}
	ephemerides.add, { $
		name: 'DSUN_AU', $
		value: solo_dist/au, $
		comment: '[AU] S/C distance from Sun'}
	ephemerides.add, { $
		name: 'AU_REF', $
		value: au, $
		comment: '[m] Assumed physical Astronomical Unit'}

	; coordinates of the S/C in the hee frame

	coord = solo_get_coords(utc, 'HEE', 'SUN') * 1000.

	ephemerides.add, { $
		name: 'HEEX_OBS', $
		value: coord[0], $
		comment: '[m] S/C Heliocentric Earth Ecliptic X'}
	ephemerides.add, { $
		name: 'HEEY_OBS', $
		value: coord[1], $
		comment: '[m] S/C Heliocentric Earth Ecliptic Y'}
	ephemerides.add, { $
		name: 'HEEZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C Heliocentric Earth Ecliptic Z'}

	; coordinates of the S/C in the hci frame

	coord = solo_get_coords(utc, 'HCI', 'SUN', velocity = veloc)
	coord = coord * 1000.
	veloc = veloc * 1000.

	ephemerides.add, { $
		name: 'HCIX_OBS', $
		value: coord[0], $
		comment: '[m] S/C Heliocentric Inertial X'}
	ephemerides.add, { $
		name: 'HCIY_OBS', $
		value: coord[1], $
		comment: '[m] S/C Heliocentric Inertial Y'}
	ephemerides.add, { $
		name: 'HCIZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C Heliocentric Inertial Z'}

	ephemerides.add, { $
		name: 'HCIX_VOB', $
		value: veloc[0], $
		comment: '[m/s] S/C Heliocentric Inertial X velocity'}
	ephemerides.add, { $
		name: 'HCIY_VOB', $
		value: veloc[1], $
		comment: '[m/s] S/C Heliocentric Inertial Y velocity'}
	ephemerides.add, { $
		name: 'HCIZ_VOB', $
		value: veloc[2], $
		comment: '[m/s] S/C Heliocentric Inertial Z velocity'}

	; coordinates of the S/C in the hae frame

	coord = solo_get_coords(utc, 'HAE', 'SUN')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'HAEX_OBS', $
		value: coord[0], $
		comment: '[m] S/C Heliocentric Aries Ecliptic X'}
	ephemerides.add, { $
		name: 'HAEY_OBS', $
		value: coord[1], $
		comment: '[m] S/C Heliocentric Aries Ecliptic Y'}
	ephemerides.add, { $
		name: 'HAEZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C Heliocentric Aries Ecliptic Z'}

	; coordinates of the S/C in the heeq frame

	coord = solo_get_coords(utc, 'HEQ', 'SUN')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'HEQX_OBS', $
		value: coord[0], $
		comment: '[m] S/C Heliocentric Earth Equatorial X'}
	ephemerides.add, { $
		name: 'HEQY_OBS', $
		value: coord[1], $
		comment: '[m] S/C Heliocentric Earth Equatorial Y'}
	ephemerides.add, { $
		name: 'HEQZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C Heliocentric Earth Equatorial Z'}

	; coordinates of the S/C in the gse frame

	coord = solo_get_coords(utc, 'GSE', 'EARTH')
	coord = coord * 1000.

	ephemerides.add, { $
		name: 'GSEX_OBS', $
		value: coord[0], $
		comment: '[m] S/C Geocentric Solar Ecliptic X'}
	ephemerides.add, { $
		name: 'GSEY_OBS', $
		value: coord[1], $
		comment: '[m] S/C Geocentric Solar Ecliptic Y'}
	ephemerides.add, { $
		name: 'GSEZ_OBS', $
		value: coord[2], $
		comment: '[m] S/C Geocentric Solar Ecliptic Z'}

	; light travel times and radial velocity of the S/C

	earth_time = get_light_time(utc, 'EARTH', 'SUN')
	sun_time = get_light_time(utc, 'SOLO', 'SUN', radvel = radvel)
	tdel = earth_time - sun_time

	ephemerides.add, { $
		name: 'OBS_VR', $
		value: radvel, $
		comment: '[m/s] Radial velocity of S/C relative to Sun'}
	ephemerides.add, { $
		name: 'EAR_TDEL', $
		value: tdel, $
		comment: '[s] Time(Sun to Earth) - Time(Sun to S/C)'}
	ephemerides.add, { $
		name: 'SUN_TIME', $
		value: sun_time, $
		comment: '[s] Time(Sun to S/C)'}

	; corrections of the acquisition date

	utc = header.date_beg

	jul_utc = date_conv(utc, 'julian')
	date_earth = date_conv(jul_utc + tdel / 86400.d0, 'fits')
	date_sun = date_conv(jul_utc - sun_time / 86400.d0, 'fits')

	ephemerides.add, { $
		name: 'DATE_EAR', $
		value: date_earth, $
		comment: '[UTC] Start time of observation, corrected to Earth'}
	ephemerides.add, { $
		name: 'DATE_SUN', $
		value: date_sun, $
		comment: '[UTC] Start time of observation, corrected to Sun'}

	return, ephemerides
end
