class artecs_map :
   """
A class to handle a temperature map from ARTECS.

It reads the artecs_map from a fits file.

   >>> AMAP=artecs_map("artecs_map.fits")

Keywords in the map are returned as members of the class or using the method "parameter".

Example:

   >>> AMAP.temp 

returns the temperature map,

   >>> AMAP.parameter('temp')
   
returns the temperature map.


   >>> AMAP.shape

returns the shape of the map (rows, columns)


   """
   @property 
   def filename(self) :
      """ parameter: the filename """
      return self._filename
   
   @property 
   def p(self) :
      """ the fits file pointer, normally closed.
      """
      return self._p
   
   @property 
   def lst_lat(self) :
      """ list of latitude bands """
      return self._lst_lat
   
   @property 
   def lst_year(self) :
      """ list of time intervals """
      return self._lst_year

   @property 
   def shape(self) :
      """ the shape of the 2D maps """
      return self._shape
   
   @property 
   def TMGLOB(self) :
      """ the mean surface temperature """
      return self._TMGLOB
   
   def __init__(self,filename,closeFits=True,verbose=False) :
      """ To instantiate the class pass the filename from which to load the Temperature map
      
      >Keywords: 
      
      >>`verbose`: if True verbose output
      
      >>`closeFits`: if True the fits file is closed after reading
      """
      import numpy
      try :
         import pyfits
      except :
         from astropy.io import fits as pyfits
      self._filename=filename
      self._p=pyfits.open(filename)
      self._key=[]
      self._value=[]
      self._descr=[]
      mkd=True
      for cc in self.p[0].header.cards :
         if cc[0]!='COMMENT' :
            self._key.append(cc[0])
            self._value.append(cc[1])
            self._descr.append(cc[2])
         else :   
            if mkd :
               self._key.append(cc[0])
               self._value.append('\n'.join(self.p[0].header['COMMENT']))
               self._descr.append(None)
               mkd=False
      
      #: N of year intervals
      self.N=self.parameter('N')
      
      #: NS number of latitude bands
      self.NS=self.parameter('NS')
      
      #: the shape
      self._shape=(self.NS,self.N)
     
      self.year=self.p[1].data['year']
      self.lat=self.p[1].data['lat']
      self.temp=self.p[1].data['temp']      
      self.year.shape=self.shape
      self.lat.shape=self.shape
      self.temp.shape=self.shape
      
      #: the mean surfscr temperature
      self._TMGLOB=self.temp.mean()

      self._lst_lat=self.lat[0]
      self._lst_year=self.year[:,0].T

      if verbose :
         print('Successfully opened and readed ',filename)
         
      if closeFits :
         self._p.close()
         if verbose :
            print('fits file closed ',filename)
         
      
   def keys(self,maps=False) : 
      """
returns the list of quantities in the map
      
Keywords: 

>`maps` (default False), if True returns just the list of of elements which are 2D maps
      """
      if maps : 
         return ['year','lat','temp']
      else :
         return self._key
   def has_key(self,key) : 
      """True if required `key` is in the map"""
      return key in self._key
   def parameter(self,key) :
      """returns a parameter from the fits file from its `key'"""
      return self._value[self._key.index(key)]
   def description(self,key) :
      """returns the description of a parameter in the fits file from its `key'"""
      return self._descr[self._key.index(key)]
   #def bilinear_interpolation(self,lat,year) :
      #"""returns bilinear interpolation of the map (not implemeted yet) """
      #raise Exception(
