__description__="""
M.Maris - 2024 Apr 19 - 

Definition of relevant physical constants in MKS and a number of usefull units transforms 

The MKS object is a table of physical constants 
"""

class __physical_parameters_mks(object) :
   @property
   def table(self) :
      return self._table
   @property
   def source(self) :
      return "https://pdg.lbl.gov/2023/reviews/contents_sports.html"
   def __init__(self) :
      """ from https://pdg.lbl.gov/2023/reviews/contents_sports.html """
      from io import StringIO
      import pandas
      import numpy as np
      # the table
      self._table=pandas.read_csv(StringIO("""
definition                        |                    latex_macro|  latex_name|       symbol| value                | units   | comment
speed of light in vacuum          |                              c|     speedL |            c| 299792458.           | m/s     |
Planck constant                   |                              h|     PConst |            h| 6.62607015e-34       | J s     | also J/Hz  
reduced Planck constant           |                           hcut|       hbar |         hbar| 1.0544571817e-34     | J s     | also J/Hz is h/2pi
Avogadro Number                   |              N_{\\mathrm{avo}}|       Navo |         Navo| 6.02214076e23        | 1/mol   |
Boltzman constant                 |                k_{\\mathrm{B}}|         kB |           kB| 1.380649e-23         | J/K     |
Universal gas constant            |                          R_{*}|       Rgas |         Rgas| 8.31446261815324     | J/K/mol | defined as kB*Navo
molar_volume                      |              V_{\\mathrm{mol}}|     MolVol |         Vmol| 22.41396954e-3       | m3/mol  | kB(273.15 K)/(101 325 Pa)
standard atmosphere pressure      |              p_{\\mathrm{atm}}|       patm |         patm| 101325.              | pa      | 
zero Celsius                      |      T_{0^{\\circ}\\mathrm{C}}|      TzCel |          T0C| 273.15               | K       |
ev in joule                       |       \\mathrm{J}/\\mathrm{ev}|        Jev |         J_ev| 1.6021766339999e-19  | J/ev    |  
rad2sed                           |    \\mathrm{rad}/\\mathrm{sed}|   radTosed |      rad2sed| 4.191690043903363e-08| rad s/m | 2*pi/c
Stefan-Boltzman Constant          |                        \\sigma|    sigmaSB |      sigmaSB| 5.670374419e-8       | W/m2/K4 | 
photons SB constant               |       \\sigma_{\\mathrm{phot}}|sigmaSBphot |  sigmaSBphot| 2.5247846571310243e-3| micro mol/m2/s/K3| 
standard solar temperature        |                     T_{\\odot}|       Tsun |         Tsun| 5772.                | K       |
standard solar radius             |                     R_{\\odot}|       Rsun |         Rsun| 696340e3             | m       | 
standard AU                       |                    \mathrm{AU}|         AU |           AU| 149597870700.        | m       | exact
gravitational constant            |                G_{\\mathrm{N}}|         GN |           GN| 6.67430e-11          | m3/Kg/s2| 
standard gravitationa acceleration|                g_{\\mathrm{N}}|         gN |           gN| 9.80665              | m/s2    | 
parsec                            |                   \\mathrm{pc}|         pc |           pc| 3.08567758149e16     | m       | exact
Wien displacement law constant    |       \\lambda_{\\mathrm{max}}| lambdaWien |   lambdaWien| 2.897771995e-3       | m K     | h c / uWien kB
Wien displacement law in frequency|           \\nu_{\\mathrm{max}}|     nuWien |       nuWien| 9.66592693602719e-12 | K / Hz  | lambdaWien/c
Wien displacement law in b.b. x   |             u_{\\mathrm{Wien}}|      uWien |        uWien| 4.965114231744276303 |         | uWien
Black Body mean photon E coeff.   |             {\\tilde E_{\\nu}}|meanBBPhotEn| meanBBPhotEn| 3.729378749850374e-23| J / K   | sigmaSB/(sigmaSBphot*1e-6*Navo)
"""),sep='\s*\|\s*',comment='#',engine="python",index_col=None)
      #
      self._table['comment'] =np.array(['' if str(k).lower() in ['nan'] else k for k in   self._table['comment'].values])
      self._table['units'] =np.array(['' if str(k).lower() in ['nan'] else k for k in   self._table['units'].values])
      #
      for ik in range(len(self._table)) :
         line=self._table.iloc[ik]
         self.__dict__[line.symbol]=line.value
   def keys(self) :
      return self._table['symbol'].values
   def __len__(self) :
      return len(self._table)
   def __getitem__(self,this) :
      if type(this) is type('') :
         try :
            return self.__dict__[this]
         except :
            raise Exception("Symbol %s not found"%str(this),"")
      else :
         try :
            return self._table.iloc[this]
         except :
            raise Exception("Symbol %s not found"%str(this),"")
   def latex(self,this) :
      """returns a latex definition for the given constant"""
      if type(this) is type('') :
         line=self._table.query('symbol=="'+this+'"')
         if line is None :
            raise Exception("Symbol %s not found"%str(this),"")
         line=line.iloc[0]
      else :
         try :
            line=self._table.iloc[line]
         except :
            raise Exception("Symbol %s not found"%str(this),"")
      out='\\def'+'\\'+line.latex_name+'{'+line.latex_macro+'}'+' % '+line.definition+', '+line.units
      return out
   def wl2nu(self,wl) :
      """wavelength to frequency"""
      return self.c/wl
   def nu2wl(self,nu) :
      """frequency to wavelength"""
      return self.c/nu
   def nu2E(self,nu) :
      """energy to frequency"""
      return self.h*nu
   def E2nu(self,E) :
      """frequency to energy"""
      return E/self.h
   def wl2E(self,wl) :
      """wavelength to energy"""
      return self.h*self.c/wl
   def E2wl(self,E) :
      """energy to wavelength"""
      return self.c*self.h/E
   def ev2J(self,ev) :
      """ ev/J"""
      return ev*self.J_ev
   def J2ev(self,J) :
      """ J/ev"""
      return J/self.J_ev
   def radiance2sed(self,radiance) :
      """ radiance -> spectral_energy_density"""
      return self.rad2sed*radiance
   def sed2radiance(self,sed) :
      """ spectral_energy_density to radiance """
      return sed/self.rad2sed
   def nu2bbx(self,nu,T) :
      """ nu and T to black body x=nu*h/kB*T """
      return nu*self.h/(T*self.kB)
   def bbx2nu(self,x,T) :
      """ black body x: x=nu*h/kB*T, to nu for given T """
      return x*T*self.kB/self.h
   def wl2bbx(self,wl,T) :
      """ wl and T to black body x=c*h/kB*T*wl """
      return self.c*self.h/(T*self.kB*wl)
   def bbx2wl(self,x,T) :
      """ black body x: x=c*h/kB*T*wl, to wl for given T """
      return self.c*self.h/(T*self.kB*x)
   def bbMeanPhotonEnergy(self,T) :
      """ black body mean photon energy for a given T"""
      return self.meanBBPhotEn*T
MKS=__physical_parameters_mks()

def wl2nu(wl) :
   """wavelength to frequency"""
   return MKS.c/wl
def nu2wl(nu) :
   """frequency to wavelength"""
   return MKS.c/nu
def nu2E(nu) :
   """energy to frequency"""
   return MKS.h*nu
def E2nu(E) :
   """frequency to energy"""
   return E/MKS.h
def wl2E(wl) :
   """wavelength to energy"""
   return MKS.h*MKS.c/wl
def E2wl(E) :
   """energy to wavelength"""
   return MKS.c*MKS.h/E
def ev2J(ev) :
   """ ev/J"""
   return ev*MKS.J_ev
def J2ev(J) :
   """ J/ev"""
   return J/MKS.J_ev
def radiance2sed(radiance) :
   """ radiance -> spectral_energy_density"""
   return MKS.rad2sed*radiance
def sed2radiance(sed) :
   """ spectral_energy_density to radiance """
   return sed/MKS.rad2sed

def nu2bbx(nu,T) :
   """ nu and T to black body x=nu*h/kB*T """
   return nu*MKS.h/(T*MKS.kB)

def bbx2nu(x,T) :
   """ black body x: x=nu*h/kB*T, to nu for given T """
   return x*T*MKS.kB/MKS.h

def wl2bbx(wl,T) :
   """ wl and T to black body x=c*h/kB*T*wl """
   return MKS.c*MKS.h/(T*MKS.kB*wl)

def bbx2wl(x,T) :
   """ black body x: x=c*h/kB*T*wl, to wl for given T """
   return MKS.c*MKS.h/(T*MKS.kB*x)

def bbMeanPhotonEnergy(T) :
   """ black body mean photon energy for a given T"""
   return MKS.meanBBPhotEn*T

