#!/usr/bin//python
import time
import datetime
import struct
import array
import os.path
import sys

import numpy as np

from juldate import juldate as _juldate
from juldate import caldate as _caldate

"""
  Module to handle different time scales used by Planck, in particular:

  UTC in ISO and ZULU format by using classes ISO, ZULU

  UTC in seconds by using class UTCsec

  TAI, PsudoOBT, SCET by using the classes TAI, OBT, SCET

  it uses Python modules: time and datetime

  Python ASSUMES UTCsec starts from 1970 Jan 01, 01:00:00.

  Conversions are described by the following conversion ring:

                        ZULU ---  ISO
                          \        /
                          JulianDay ------- PseudoOD
                         /         \
                        TAI     PseudoOBT
                         \         /
                         SCET - UTCsec
                          
  where /,\,- denotes interconversion possibility.

  PseudoOBT is OBT calculate ASSUMING 
    -- onboard clock properly set at zero time
    -- onboard clock advancing at a constant rate of 1sec/sec
    -- onboard clodk without drifts.

  Differences between PseudoOBT and true OBT are expected to be of the order of some second.

  Note that since Python Datetime class assumes use of UTCsec, the only method
  returnin a Datetime object is UTCsec.datetime() use to convert an UTCsec into
  a Datetime.

  PseudoOD is calculate ASSUMING to count the number of days after the launch epoch.
  So PseudoOD is interconnected just to JulianDay, since PseudoOD is nothing else than 
  Julian Day with a different starting time.
  
  WARNING OD DEPRECATED:
     To avoid confusion with the true OD (Level1 OD) which can not be computed but which can be
  tabulated from AHF the class OD should be avoided. The class OD and JuliandDay-OD conversion
  methods are DEPRECATED. 
  
  USE PseudoOD instead.

  Author: 
    Michele Maris, Issue 1.0, 30 Mar 2010
                   Issue 1.1, 14 May 2010 - added management of ODs
                   Issue 1.2, 25 Jun 2010 - corrected OD zero point
                                            OD is now deprecated USE PseudoOD instead
                                            added the possibility to use it as a command lines tool
"""

class _TIME_SCALE_BASE() :
   """ base class to handle times: private class
       for each base: TAI, SCET, PseudoOBT, UTC defines:
       zero point (example ZERO_TAI) and if needed TICKS number of tickes per second

       It provides methods to recover such quantities,
       a __str__ method and a set method.

       Basic representation of times is Julian Dates:
   """
   ZERO_MJD = _juldate((1858,11,17,12,0,0))
   ZERO_TAI = _juldate((1958,1,1,0,0,0,0))
   ZERO_SCET = _juldate((1958,1,1,0,0,0,0))
   ZERO_UTC = _juldate((1970,1,1,0,0,0,0))
   ZERO_PseudoOBT = _juldate((1958,1,1,0,0,0,0))

   #ZERO_OD = _juldate((2009,5,14,10,13,00))

   ZERO_OD = _juldate((2009,5,13,13,12,00))

   TICKS_SCET_SEC = 1e6
   TICKS_PseudoOBT_SEC = float(2**16)

   TICKS_PseudoOBT_DAY = float(86400)*TICKS_PseudoOBT_SEC
   TICKS_SCET_DAY = float(86400)*TICKS_SCET_SEC
   TICKS_UTC_DAY = float(86400)
   TICKS_TAI_DAY = float(86400)

   def __init__(self,v=0.) :
      """ creator accepts scalars, strings, and lists 
          
          scalar (float, int, long): saves the value as double

          list : assumes a tuple of type: (YYYY, MM, DD, HH, MM, SS_DEC) and converts in Julian Day

          string : assumes a string of type: "YYYY-MM-DDTHH:MM:SS.DEC" and converts in Julian Day
                   if empty is equal to put v = 0
          use method set(v) to change value to a scalar, time() to return value
      """
      if self._isNumericScalar(v) : 
         self.v = np.float64(v)
         return

      if type(v) == type(()) or type(v) == type([]) :
         if len(v) == 3 :
            self.v = _juldate((v[0],v[1],v[2],0.,0.,0.))
            return
         if len(v) == 5 :
            self.v = _juldate((v[0],v[1],v[2],v[3],v[4],0.))
            return
         else :
            self.v = _juldate(v)
            return
     
      if type(v) == type('') : 
         if v == '' :
            self.v = 0
            return
         else :
            self.v = _juldate((float(v[0:4]), float(v[5:7]), float(v[8:10]), float(v[11:13]), float(v[14:16]), float(v[17:len(v)])))
            return

      print "unrecognized input type"
      return

   def __str__(self) :
       return str(self.v)

   def _isNumericScalar(self,v) :
      tv = type(v)
      return tv == type(0.) or tv == type(0) or tv == type(long(0)) or tv == type(np.float64(0))

   def _tuple2isostring(self,tp) :
      """ converts a tuple to an ISO time string """
      if len(tp) == 6 :
         return "%04d-%02d-%02dT%02d:%02d:%010.7f" % tp
      return ""

   def set(self,v) :
      self.v = float(v)

   def time(self) :
       return self.v

   def kind(self,test='') :
       return False

   def zero_tai(self) :
       return self.__class__.ZERO_TAI

   def zero_utc(self) :
       return self.__class__.ZERO_UTC

   def zero_pseudoObt(self) :
       return self.__class__.ZERO_PseudoOBT

   def zero_od(self) :
       return self.__class__.ZERO_OD

   def step_scet_sec(self) :
       return 1./self.__class__.TICKS_SCET_SEC

   def step_pseudoObt_sec(self) :
       return 1./self.__class__.TICKS_PseudoOBT_SEC

class _TIME_SCALE_PART(_TIME_SCALE_BASE) :
    """ base class to handle times divided into two parts: 
        - a coarse part (long)
        - a fine part (double)
    """
    def __init__(self,coarse=0,fine=-1.) :
       if fine < 0 :
          self.coarse = floor(coarse)
          self.fine   = coarse-self.coarse
       else :
          self.coarse = coarse
          self.fine   = fine

    def tuple(self) :
       """ returns a tuple """
       return (self.coarse, self.fine)

    def __str__(self,sept=':') :
       return "%s%s%s" % (self.coarse,sept,self.fine)

    def getTime(self,fine_ticks_sec=1.) :
       """ getTime([ticks_fine=1.]) combines coarse and fine part into a float 
           getTime = float(coarse)+self_fine/float(fine_ticks_sec)
       """
       return self.coarse+self.fine/float(fine_ticks_sec)

class _TIME_SCALE_STRING(_TIME_SCALE_BASE) : 
   """ base class to handle times in form of strings """
   def __init__(self,v='') :
      self.v=str(v)

   def __str__(self) :
       return str(self.v)

   def set(self,v) :
      self.v=str(v)


class CalendarDate(_TIME_SCALE_BASE) :
   def __init__(self,v) :
      if type(v) == type(()) or type(v) == type([]) :
         if len(v) == 3 :
            self.v = (long(v[0]),long(v[1]),long(v[2]),long(0),long(0),float(0.))
            return
         if len(v) == 5 :
            self.v = (long(v[0]),long(v[1]),long(v[2]),long(v[3]),long(v[4]),float(0.))
            return
         else :
            self.v = v
            return
         print "unrecongnized ntuple format ",v
         return 

      if type(v) == type('') : 
         if v == '' :
            self.v = (long(0), long(0), long(0), long(0), long(0), float(0.))
            return
         else :
            self.v = ((long(v[0:4]), long(v[5:7]), long(v[8:10]), long(v[11:13]), long(v[14:16]), float(v[17:len(v)])))
            return
         print "unrecognized string format ",v
         return 

      if type(v) == type(JulianDay(0.)) :
         self.v=_caldate(v.v)

      print "unrecognized format ",v
      return 

   def kind(self,test='') :
      """ returns kind of object """
      if test == '' :
          return 'CalendarDate'
      return test == 'CalendarDate'

   def julianday(self) :
      """ conversion to julian day """
      return JulianDay(self.v)

   def iso(self) :
      """ Conversion in ISO format """
      return ISO(self._tuple2isostring(self.v))

   def zulu(self) :
      """ Conversion in ISO format """
      return ISO(self._tuple2isostring(self.v)).zulu()

   def tuple(self) :
      """ returns the tuple """
      return self.v

class ISO(_TIME_SCALE_STRING) : 
   """ class to handle times in form of ISO strings YYYY-MM-DDTHH:MM:SS   """
   def kind(self,test='') :
      """ returns kind of object """
      if test == '' :
          return 'ISO'
      return test == 'ISO'

   def julianday(self) :
      """ returns Julian Day """
      return JulianDay(self.v)

   def calendardate(self) :
      """ returns the calendar date """
      return CalendarDate(self.v)

   def tuple(self) : 
      """ returns the tuple for the date"""
      return (long(self.v[0:4]), long(self.v[5:7]), long(self.v[8:10]), long(self.v[11:13]), long(self.v[14:16]), float(self.v[17:len(self.v)]))

   def zulu(self) :
      """ returns Zulu """
      return ZULU(self.v+'Z')

class ZULU(_TIME_SCALE_STRING) : 
   """ class to handle times in form of ISO ZULU strings YYYY-MM-DDTHH:MM:SSZ  """
   def __init__(self,v='Z') :
      _TIME_SCALE_STRING.__init__(self,v)
      if not (self.v[len(self.v)-1] == 'Z') :
         self.v=self.v+'Z'

   def kind(self,test='') :
      """ returns kind of object """
      if test == '' :
          return 'ZULU'
      return test == 'ZULU'

   def julianday(self) :
      """ returns Julian Day """
      return JulianDay(self.v[0:(len(self.v)-1)])

   def calendardate(self) :
      """ returns the calendar date """
      return self.iso().calendardate()

   def tuple(self) : 
      """ returns the tuple for the date"""
      return (long(self.v[0:4]), long(self.v[5:7]), long(self.v[8:10]), long(self.v[11:13]), long(self.v[14:16]), long(self.v[17:(len(self.v)-1)]))

   def iso(self) :
      """ returns Zulu """
      return ISO(self.v[0:(len(self.v)-1)])

class JulianDay(_TIME_SCALE_BASE) :
   """ class to handle JulianDay """
   def kind(self,test='') :
      if test == '' :
         return 'JulianDay'
      return test == 'JulianDay'

   def pseudoObt(self) :
      return PseudoOBT((self.v-self.__class__.ZERO_PseudoOBT)*self.__class__.TICKS_PseudoOBT_DAY)

   def utcsec(self) :
      return UTCsec((self.v-self.__class__.ZERO_UTC)* self.__class__.TICKS_UTC_DAY)

   def scet(self) :
      return SCET((self.v-self.__class__.ZERO_SCET)* self.__class__.TICKS_SCET_DAY)

   def tai(self) :
      return TAI((self.v-self.__class__.ZERO_TAI)* self.__class__.TICKS_TAI_DAY)

   def calendardate(self) :
      return CalendarDate(_caldate(self.v))

   def tuple(self) :
      return CalendarDate(_caldate(self.v)).tuple()

   def iso(self) : 
       return ISO("%04d-%02d-%02dT%02d:%02d:%010.7f" % _caldate(self.v))

   def zulu(self) : 
       return ZULU("%04d-%02d-%02dT%02d:%02d:%010.7f" % _caldate(self.v))

   def od(self) :
      return OD(self.v - self.__class__.ZERO_OD)

   def pseudoOd(self) :
      return PseudoOD(self.v - self.__class__.ZERO_OD)

class OD(_TIME_SCALE_BASE) :
   """ class to handle OD 
       (a different way to handle a Julian Day)
       Note, the only operations allowed are: JulianDay <-> OD conversions
       
       DEPRECATED 
          This class is deprecated, use PseudoOD instead
   """
   def __init__(self,v=0.) :
      _TIME_SCALE_BASE.__init__(self,v)
      if self.v < 0. :
         print "Waning: negative od"

   def kind(self,test='') :
      if test == '' :
         return 'OD'
      return test == 'OD'

   def julianday(self) :
      return JulianDay( self.v+self.__class__.ZERO_OD)

   def pseudoObt(self) :
      return 

   def utcsec(self) :
      return 

   def scet(self) :
      return 

   def tai(self) :
      return 

   def calendardate(self) :
      return 

   def tuple(self) :
      return 

   def iso(self) : 
       return 

   def zulu(self) : 
       return 

class PseudoOD(_TIME_SCALE_BASE) :
   """ class to handle PseudoOD 
       (a different way to handle a Julian Day)
       Note, the only operations allowed are: JulianDay <-> PseudoOD conversions
   """
   def __init__(self,v=0.) :
      _TIME_SCALE_BASE.__init__(self,v)
      if self.v < 0. :
         print "Waning: negative od"

   def kind(self,test='') :
      if test == '' :
         return 'PseudoOD'
      return test == 'PseudoOD'

   def julianday(self) :
      return JulianDay( self.v+self.__class__.ZERO_OD)

   def pseudoObt(self) :
      return 

   def utcsec(self) :
      return 

   def scet(self) :
      return 

   def tai(self) :
      return 

   def calendardate(self) :
      return 

   def tuple(self) :
      return 

   def iso(self) : 
       return 

   def zulu(self) : 
       return 

class PseudoOBT(_TIME_SCALE_BASE) :
   """ class to handle PseudoOBT : in 2^-16 seconds since 1958 Jan 01, 00:00:00 
       
       PseudoOBT is OBT calculate ASSUMING 
         -- onboard clock properly set at zero time
         -- onboard clock advancing at a constant rate of 1sec/sec
         -- onboard clodk without drifts.

       Differences between PseudoOBT and true OBT are expected to be of the order of some second.
   """
   def __init__(self,v=0.) :
      _TIME_SCALE_BASE.__init__(self,v)
      # if v is not a numeric scalar then v has been coded as a JD and it has to be converted
      if not self._isNumericScalar(v) :
         self.v = (self.v - self.ZERO_PseudoOBT)*self.TICKS_PseudoOBT_DAY
         
   def kind(self,test='') :
      if test == '' :
         return 'PseudoOBT'
      return test == 'PseudoOBT'

   def utcsec(self) :
      return UTCsec((self.v/self.__class__.TICKS_PseudoOBT_DAY+self.__class__.ZERO_PseudoOBT-self.__class__.ZERO_UTC)* self.__class__.TICKS_UTC_DAY)

   def scet(self) :
      return SCET((self.v/self.__class__.TICKS_PseudoOBT_DAY+self.__class__.ZERO_PseudoOBT-self.__class__.ZERO_SCET)*self.__class__.TICKS_SCET_DAY)

   def tai(self) :
      return TAI( (self.v/self.__class__.TICKS_PseudoOBT_DAY+self.__class__.ZERO_PseudoOBT-self.__class__.ZERO_TAI)*self.__class__.TICKS_TAI_DAY)

   def julianday(self) :
      return JulianDay( (self.v/self.__class__.TICKS_PseudoOBT_DAY+self.__class__.ZERO_PseudoOBT))

   def iso(self) :
      return self.julianday().iso()
    
   def zulu(self) :
      return self.julianday().zulu()


class UTCsec(_TIME_SCALE_BASE) :
   """ class to handle UTC times in seconds since 1970 Jan 01, 00:00:00 """
   def __init__(self,v=0.) :
      _TIME_SCALE_BASE.__init__(self,v)
      # if v is not a numeric scalar then v has been coded as a JD and it has to be converted
      if not self._isNumericScalar(v) :
         self.v = (self.v - self.ZERO_UTC)*86400

   def kind(self,test='') :
      if test == '' :
          return 'UTCsec'
      return test == 'UTCsec'

   def datetime(self) :
      """ Returns a Datetime object from UTCsec """
      return datetime.datetime.fromtimestamp(self.v,None)

   def scet(self) :
      return SCET( (self.v/self.__class__.TICKS_UTC_DAY+self.__class__.ZERO_UTC-self.__class__.ZERO_SCET) * self.__class__.TICKS_SCET_DAY )

   def tai(self) :
      return TAI( (self.v/self.__class__.TICKS_UTC_DAY+self.__class__.ZERO_UTC-self.__class__.ZERO_TAI)*self.__class__.TICKS_TAI_DAY )

   def julianday(self) :
      return JulianDay( (self.v/self.__class__.TICKS_UTC_DAY+self.__class__.ZERO_UTC))

   def pseudoObt(self) :
      return PseudoOBT((self.v/self.__class__.TICKS_UTC_DAY+self.__class__.ZERO_UTC-self.__class__.ZERO_PseudoOBT)*self.__class__.TICKS_PseudoOBT_DAY)

   def iso(self) :
      return self.julianday().iso()
    
   def zulu(self) :
      return self.julianday().zulu()


class SCET(_TIME_SCALE_BASE) :
   """ class to handle SCET : in microseconds since 1958 Jan 01, 00:00:00 """
   def __init__(self,v=0.) :
      _TIME_SCALE_BASE.__init__(self,v)
      # if v is not a numeric scalar then v has been coded as a JD and it has to be converted
      if not self._isNumericScalar(v) :
         self.v = (self.v - self.__class__.ZERO_SCET)*self.__class__.TICKS_SCET_DAY

   def kind(self,test='') :
       if test == '' :
           return 'SCET'
       return test == 'SCET'

   def tai(self) :
      return TAI( (self.v/self.__class__.TICKS_SCET_DAY+self.__class__.ZERO_SCET-self.__class__.ZERO_TAI) * self.__class__.TICKS_TAI_DAY )

   def julianday(self) :
      return JulianDay( (self.v/self.__class__.TICKS_SCET_DAY+self.__class__.ZERO_SCET))

   def pseudoObt(self) :
      return PseudoOBT((self.v/self.__class__.TICKS_SCET_DAY+self.__class__.ZERO_SCET-self.__class__.ZERO_PseudoOBT)*self.__class__.TICKS_PseudoOBT_DAY)

   def utcsec(self) :
      return UTCsec( (self.v/self.__class__.TICKS_SCET_DAY+self.__class__.ZERO_SCET-self.__class__.ZERO_UTC)*self.__class__.TICKS_UTC_DAY )

   def iso(self) :
      return self.julianday().iso()
    
   def zulu(self) :
      return self.julianday().zulu()



class TAI(_TIME_SCALE_BASE) :
   """ class to handle TAI : same than UTC but from 1958 Jan 01, 00:00:00 """
   def __init__(self,v=0.) :
      _TIME_SCALE_BASE.__init__(self,v)
      # if v is not a numeric scalar then v has been coded as a JD and it has to be converted
      if not self._isNumericScalar(v) :
         self.v = (self.v - self.__class__.ZERO_TAI)*self.__class__.TICKS_TAI_DAY

   def kind(self,test='') :
      if test == '' :
         return 'TAI'
      return test == 'TAI'

   def julianday(self) :
      return JulianDay( (self.v/self.__class__.TICKS_TAI_DAY+self.__class__.ZERO_TAI))

   def pseudoObt(self) :
      return PseudoOBT((self.v/self.__class__.TICKS_TAI_DAY+self.__class__.ZERO_TAI-self.__class__.ZERO_PseudoOBT)*self.__class__.TICKS_PseudoOBT_DAY)

   def utcsec(self) :
      return UTCsec( (self.v/self.__class__.TICKS_TAI_DAY+self.__class__.ZERO_TAI-self.__class__.ZERO_UTC)*self.__class__.TICKS_UTC_DAY )

   def scet(self) :
      return SCET( (self.v/self.__class__.TICKS_TAI_DAY+self.__class__.ZERO_TAI-self.__class__.ZERO_SCET) * self.__class__.TICKS_SCET_DAY )

   def iso(self) :
      return self.julianday().iso()
    
   def zulu(self) :
      return self.julianday().zulu()



def _TEST_time_support() :
   """ test and examples """

   x = JulianDay(2400000.5)
   print
   print x.kind(),x, "input ad number "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.calendardate().kind()),x.calendardate()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()

   x = JulianDay((1858,11,17,0,0,0))
   print
   print x.kind(),x,"input as tuple "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.calendardate().kind()),x.calendardate()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()

   x = JulianDay("1858-11-17T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.calendardate().kind()),x.calendardate()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()

   x = PseudoOBT(0)
   print
   print x.kind(),x,"input as number  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()


   x = PseudoOBT((1958,1,1,0,0,0))
   print
   print x.kind(),x,"input as tuple  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()


   x = PseudoOBT("1958-01-01T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()


   x = UTCsec(0)
   print
   print x.kind(),x,"input as number  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()


   x = UTCsec((1970,1,1,0,0,0))
   print
   print x.kind(),x,"input as tuple  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()


   x = UTCsec("1970-01-01T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()


   x = SCET(0)
   print
   print x.kind(),x,"input as number  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()


   x = SCET((1970,1,1,0,0,0))
   print
   print x.kind(),x,"input as tuple  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()


   x = SCET("1970-01-01T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.tai().kind()),x.tai()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()


   x = TAI(0)
   print
   print x.kind(),x,"input as number  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()


   x = TAI((1970,1,1,0,0,0))
   print
   print x.kind(),x,"input as tuple  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()


   x = TAI("1970-01-01T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.pseudoObt().kind()),x.pseudoObt()
   print " %10s -> %10s : " % (x.kind(),x.utcsec().kind()),x.utcsec()
   print " %10s -> %10s : " % (x.kind(),x.scet().kind()),x.scet()


   x = ISO("1970-01-01T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.calendardate().kind()),x.calendardate()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()

   x = ZULU("1970-01-01T00:00:00")
   print
   print x.kind(),x,"input as string  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.calendardate().kind()),x.calendardate()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()

   x = CalendarDate((1970,1,1,0,0,0.))
   print
   print x.kind(),x,"input as tuple  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()

   x = CalendarDate((2009,5,14,10,13,0.)).julianday()
   print
   print x.kind(),x,"input as scalar  "
   print " %10s -> %10s : " % (x.kind(),x.iso().kind()),x.iso()
   print " %10s -> %10s : " % (x.kind(),x.zulu().kind()),x.zulu()
 #  print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),type(x.tuple())),x.tuple()
   print " %10s -> %10s : " % (x.kind(),x.od().kind()),x.od()

   x = CalendarDate((2009,5,14,10,13,0.)).julianday().od()
   print
   print x.kind(),x,"input as scalar  "
   print " %10s -> %10s : " % (x.kind(),x.julianday().kind()),x.julianday()
   print " %10s -> %10s : " % (x.kind(),x.julianday().calendardate().kind()),x.julianday().calendardate()

# end
#_TEST_time_support()

if __name__ == '__main__' :
   # creates the obj by od database
   import os
   import sys
   import shutil
   from optparse import OptionParser

   class _CmdLine() :
      def __init__(self) :
         self.ArgList = ['in','out']
         
         cmdln = OptionParser(usage="usage: %prog [options] "+
            " ".join(self.ArgList)
            +"[options]",version='1.0 - 2010 Jun 25 - M.Maris')
         cmdln.add_option("-d","--dry-run",dest="DryRun", action = 'store_true',default=False,help='Performs a dry run')
         cmdln.add_option("-T","--test",dest="ACTION_TEST", action = 'store_true',default=False,help='run a test')
         
         (self.Options,self.Args) = cmdln.parse_args()
         
         if len(self.Args) != len(self.ArgList) and not self.Options.ACTION_TEST :
            print
            print "TIME_SUPPORT "
            cmdln.print_version()
            cmdln.print_help()
            print
            print "Allowed codes for times (not case sensitive): "
            print "   ISO, ZULU, JD, SCET, TAI, PseudoOBT, PseudoOD"
            print
            print "Example:"
            print "   > time_support.py PseudoOD=0 ISO "
            print "   2009-05-14T13:11:59.9999839"
            print "   > time_support.py PseudoOD=0 Z "
            print "   2009-05-14T13:11:59.9999839Z"
            print
            sys.exit(1)
         else :
            if len(self.Args) > 0 :
               self.IN = self.Args[0]
               self.OUT = self.Args[1]
            else :
               self.IN = ""
               self.OUT = ""
               
            if len(self.IN) == 0 :
               self.IN_TYPE = ''
               self.IN_VALUE = ''
            else :
               ll = self.IN.split('=')
               if len(ll) != 2 :
                  sys.exit(1)
               else :
                  self.IN_TYPE = ll[0]
                  self.IN_VALUE = ll[1]
            self.DryRun()
      
      def DryRun(self) :
         if self.Options.DryRun :
            print "Dry Run\nExit\n"
            sys.exit(1)
   
   
   # execution of commands
   cmdline = _CmdLine()
   if cmdline.Options.ACTION_TEST :
      _TEST_time_support()
      sys.exit(1)
   
   JD = None
   if cmdline.IN_TYPE.lower() == 'ISO'.lower() :
      JD = JulianDay(cmdline.IN_VALUE) 
      
   if cmdline.IN_TYPE.lower() == 'ZULU'.lower() :
      JD = ZULU(cmdline.IN_VALUE).julianday()
      
   if cmdline.IN_TYPE.lower() == 'JD'.lower() :
      JD = JulianDay(float(cmdline.IN_VALUE))
   
   if cmdline.IN_TYPE.lower() == 'PseudoOBT'.lower() :
      JD = PseudoOBT(float(cmdline.IN_VALUE)).julianday()
   
   if cmdline.IN_TYPE.lower() == 'TAI'.lower() :
      JD = TAI(float(cmdline.IN_VALUE)).julianday()
   
   if cmdline.IN_TYPE.lower() == 'SCET'.lower() :
      JD = SCET(float(cmdline.IN_VALUE)).julianday()
   
   if cmdline.IN_TYPE.lower() == 'PseudoOD'.lower() :
      JD = OD(float(cmdline.IN_VALUE)).julianday()

   if JD == None :
      print "Unknown input type"
      
   if cmdline.OUT.lower() == 'ISO'.lower() :
      print JD.iso()
      sys.exit(1)
   
   if cmdline.OUT.lower() == 'PseudoOBT'.lower() :
      print JD.pseudoObt()
      sys.exit(1)
   
   if cmdline.OUT.lower() == 'TAI'.lower() :
      print JD.tai()
      sys.exit(1)
   
   if cmdline.OUT.lower() == 'SCET'.lower() :
      print JD.scet()
      sys.exit(1)
   
   if cmdline.OUT.lower() == 'ZULU'.lower() :
      print JD.zulu()
      sys.exit(1)
   
   if cmdline.OUT.lower() == 'PseudoOD'.lower() :
      print JD.od()
      sys.exit(1)
      
   if cmdline.OUT.lower() == 'JD'.lower() :
      print JD.v
      sys.exit(1)
   
   print "Unknown output type"
   
