__DESCRIPTION__=""" 1.0 M.Maris - 2018 Nov - 2022 Nov 29 -"""

class wavenumber_base :
   """A class to handle a base (list) of wavenumbers at regular wavenumber intervalls

   Attributes:
      wl_min  : minimum wavelength (usually micron, immutable)
      wl_min  : maximum wavelength (usually micron, immutable)
      wn_step : step in wavenumber (usually cm-1, immutable)
      size    : number of elements in the array
      shape   : shape of the array
      wn      : the list of wavenumbers
      wl      : the list of wavelengths

   Immutable attributes cannot be changed after instantiation
   """
   @property
   def wn_step(self) :
      return self._step
   def __init__(self,xmin,xmax,wn_step,limits_are_wl=True,wl_wn_conversion_factor=1e4) :
      """ generates a base of wn for given wn_step

      Args:
         xmin,xmax : minimum, maximum wavelength (usually micron)
         wn_step   : step in wavenumber (usually cm-1)

      Keys:
         limits_are_wl : if True xmin and xmax are wavelengths otherwise are wavenumbers (default True)
         wl_wn_conversion_factor : coversion factor from wavelength to wavenumber (default 1e4 cm-1/micron)
      """
      import numpy as np
      self._step=wn_step
      #
      self._wl_wn_conversion_factor=wl_wn_conversion_factor
      #
      if limits_are_wl :
         self.wl_min=xmin
         self.wl_max=xmax
         self._min=wl_wn_conversion_factor/xmax
         self._max=wl_wn_conversion_factor/xmin
      else :
         self._min=xmin
         self._max=xmax
      #
      self._imin=int(np.floor(self._min/float(self._step)))
      self._imax=int(np.ceil(self._max/float(self._step)))
      #
      self.size=self._imax-self._imin+1
      self.shape=(self.size,)
      #
      self.wn=(np.arange(self.size)+self._imin)*float(self._step)
      #
      if limits_are_wl==False :
         self.wl_min=wl_wn_conversion_factor/xmax
         self.wl_max=wl_wn_conversion_factor/xmin
      #
      self.wl=wl_wn_conversion_factor/self.wn
   #
   def __len__(self) :
      return self.size
   #
   def project(self,wnIn,XIn) :
      """projects (interpolates) X(wn) based on wn into the current wavenumber_base
         needs tabulated input wnIn and XIn=X(wnIn)

         values of wnbase outside [wnIn.min,wnIn.max] are filled with zeroes

         if len(wnIn) == 0 : returns an array filled with zeroes
         if len(wnIn) == 1 : assumes constant X value

         Args:
            wn : list of wavenumbers
            X  : X of values X(wn)

         Returns:
           Array of X values linearly interpolated at the wn of the instantiated object
      """
      import numpy as np
      if np.isscalar(wnIn) :
         return np.zeros(len(self))+XIn
      if len(wnIn) == 0 :
         return np.zeros(len(self))
      if len(wnIn) == 1 :
         return np.zeros(len(self))+XIn[0]
      return np.interp(self.wn,wnIn,XIn,left=0.,right=0.)

