from collections import OrderedDict
import numpy as np
import pandas
import io
import time

class AppendableDict :
   """ a dictionary, whose arguments are lists which can be appended 
   
   Example: 
      >AD=AppendableDict()
      >
      >AD['line1']=1
      >AD['line2']=10
      >
      >AD['line1']=2
      >AD['line2']=20
      >
      >AD['line1']=3
      >AD['line2']=30
      >
      >print(AD['line1'])
      [1,2,3]
      
   """
   def __init__(self,comment=None) :
      """
      Keys: 
        comment : comment string
      """
      self._k=[]
      self._d={}
      self._comment="" if comment is None else comment
   def keys(self) :
      """ list of arguments """
      return self._k
   def __getitem__(self,this) :
      return self._d[this]
   def __setitem__(self,this,that) :
      if this in self.keys() :
         self._d[this].append(that)
      else :
         self.keys().append(this)
         self._d[this]=[that]
   def isempty(self) :
      """ true if empty dictionary """
      return len(self.keys()) == 0
   def same_columns_length(self) :
      """ true if all the arguments have the same lenght """
      out=np.array([len(self[k]) for k in self.keys()])
      if out.ptp()==0 : return True
      return False
   def appendDict(self,_dict) :
      """ append another dictionary """
      for k in _dict.keys() :
         self[k]=_dict[k]
   def set(self,key,value) :
      if not key in self.keys() :
         self._k.append(key)
      self._d[key]=value
   def flatten_nested_columns(self) :
      """ treats contents as nested arrays and serializes them
          beware: after this operation any appending or inserting
          of new variables/values will produce unpredictable results
      """
      for k in self._k :
         self._d[k]=np.concatenate(self._d[k])
   def to_arrayDict(self,flatten=False) :
      """ dictionary to numpy.array
          if flatten==True applies .flatten_nested_columns() method
      """
      out=OrderedDict()
      if flatten :
         for k in self.keys() :
            out[k]=np.concatenate(self[k])
      else :
         for k in self.keys() :
            out[k]=np.array(self[k])
      return out
   def to_pandas(self,flatten=False) :
      """ dictionary to pandas.DataFrame
          if flatten==True applies .flatten_nested_columns() method
      """
      out=self.to_arrayDict(flatten=flatten)
      return pandas.DataFrame(out)
   def to_csv_slow(self,csvfile,justBody=False,creator="",index_label=None,sep=',',comment=None,addEnd=False,mode='w') :
      """ dictionary to csv, slow version """
#      from collections import OrderedDict 
#      import pandas
#      import time
#      import io
      #
      Dict=self._d
      #
      headf=','.join(Dict.keys())
      #
      n=np.array([len(self[i]) for i in Dict.keys()])
      if (n.ptp()!=0) :
         raise Error("columns of different lenght ",n)
      #
      body=io.StringIO()
      for i in range(int(n[0])) :
         l=[]
         for k in Dict.keys() :
            l.append(str(Dict[k][i]))
         l=(','.join(l))+'\n'
         body.write(l)
         print(l)
      body=body.getvalue()
      #
      if justBody :
         if not csvfile is None :
            open(csvfile,mode).write(body+'\n')
         return body
      #
      out=io.StringIO()
      out.write("#\n")
      out.write("#!filename="+(csvfile if not csvfile is None else "")+'\n')
      out.write("#!created="+time.asctime()+'\n')
      out.write("#!creatore="+(creator if type(creator)==type("") else "")+'\n')
      out.write("#\n")
      if not self._comment is None :
         cc='\n'.join(['#'+k for k in self._comment.split('\n')])
         out.write(cc+'\n')
         out.write('#\n')
      if not comment is None :
         cc='\n'.join(['#'+k for k in comment.split('\n')])
         out.write(cc+'\n')
         out.write('#\n')
      out.write("#!BEGIN\n")
      out.write(headf+'\n')
      out.write(body+'\n')
      if addEnd :
         out.write("#!END\n")
      out=out.getvalue()
      if not csvfile is None :
         open(csvfile,mode).write(out)
      return out
   def to_csv(self,csvfile,justBody=False,creator="",index_label='index',sep=',',comment=None,addEnd=True,mode='w',sort_by=None) :
      """ dictionary to csv"""
#      from collections import OrderedDict 
#      import pandas
#      import time
#      import io
      #
      opd=self.to_pandas()
      if not sort_by is None:
         opd['_'+sort_by]=np.array(opd[sort_by].values)
         opd.sort_values('_'+sort_by,inplace=True)
         opd=opd.reindex()
         opd.drop('_'+sort_by,inplace=True,axis=1)
      else :
         for k in self.keys() :
            opd[k]=np.array(self[k])
      #opd=self.to_pandas()
      #
      out=io.StringIO()
      if not justBody :
         out.write("#\n")
         out.write("#!filename="+csvfile+'\n')
         out.write("#!created="+time.asctime()+'\n')
         out.write("#!creator="+(creator if type(creator)==type("") else "")+'\n')
         out.write("#\n")
         if not self._comment is None :
            cc='\n'.join(['#'+k for k in self._comment.split('\n')])
            out.write(cc+'\n')
            out.write('#\n')
         if not comment is None :
            cc='\n'.join(['#'+k for k in comment.split('\n')])
            out.write(cc+'\n')
            out.write('#\n')
         out.write("#!BEGIN\n")
      opd.to_csv(out,sep=',',index=(not index_label is None),index_label=index_label)
      #
      if not justBody and addEnd :
         out.write("#!END\n")
      #
      #removes empty lines, if any
      out=[k.strip() for k in out.getvalue().split('\n') if k.strip()!='']
      #
      #removes header if just body is needed
      if justBody :
         out=out[1:]
      #
      out='\n'.join(out)+'\n'
      if not csvfile is None :
         open(csvfile,mode).write(out)
      return out

